const jsonPunctRe = /[[\]{},:]/;
const jsonNumberCharRe = /[-+eE0-9.]/;
const jsonWhitespaceRe = /[ \t\r\n]/;

const literals = ['true', 'false', 'null'];
const partialLiteralRe = /^(?:t(?:ru?)?|f(?:a(?:ls?)?)?|n(?:ul?)?)$/; // t|tr|tru|f|fa|fal|fals|n|nu|nul

const YIELD_LIMIT = 2000; // yield roughly every this many characters; without this there are performance issues on large inputs
// The number 2000 was obtained experimentally -- 1000 and 2000 and 4000 are roughly the same; 500 is much slower; 16000 is much slower

function isNumber(value) {
  return !!value && jsonNumberCharRe.test(value.charAt(0));
}

function isString(value) {
  return value.startsWith('"');
}






class TokenizingTransformer {
  position = 0; // the position of startIndex in the entire stream
  startIndex = 0; // the index within the current buffer up to which has been tokenized
  buffer = '';
  endString = 0;
  endNumber = 0;

  async transform(chunk, controller) {
    for (let i = 0; i < chunk.length; i += YIELD_LIMIT) {
      const bit = chunk.substring(i, i + YIELD_LIMIT);
      this.buffer += bit;
      this.tokenize(controller, false);
      if (i + YIELD_LIMIT < chunk.length) await new Promise((r) => setTimeout(r, 0));
    }
  }

  flush(controller) {
    this.tokenize(controller, true);
  }

  enqueue(controller, size) {
    const value = this.buffer.substring(this.startIndex, this.startIndex + size);
    controller.enqueue({ value, position: this.position });
    this.startIndex += size;
    if (this.startIndex >= 8192) {
      this.buffer = this.buffer.substring(this.startIndex);
      this.startIndex = 0;
    }
    this.position += size;
    this.endString = 0;
    this.endNumber = 0;
  }

  tokenize(controller, final) {
    while (this.buffer.length > this.startIndex) {
      if (this.endString === 0 && this.endNumber === 0) {
        this.removeInitialWhitespace();
        if (this.buffer.length === this.startIndex) break;
      }
      const tokenSize = this.findOneToken();
      if (tokenSize > 0) {
        this.enqueue(controller, tokenSize);
        continue;
      } else if (this.endString > 0) {
        if (!final) break;
        controller.error(new SyntaxError('Unterminated string in JSON at position ' + (this.position + this.endString)));
        return;
      } else if (this.endNumber > 0) {
        if (!final) break;
        this.enqueue(controller, this.endNumber);
      } else {
        if (!final) {
          if (this.buffer.length - this.startIndex <= 4) {
            if (partialLiteralRe.test(this.buffer.substring(this.startIndex))) break;
          }
        }
        // garbage
        controller.error(new SyntaxError('Invalid JSON token at position ' + this.position));
        return;
      }
    }
  }

  findOneToken() {
    if (this.endString === 0 && this.endNumber === 0) {
      const ch = this.buffer.charAt(this.startIndex);
      if (jsonPunctRe.test(ch)) {
        return 1;
      }
      for (const literal of literals) {
        if (this.buffer.startsWith(literal, this.startIndex)) {
          return literal.length;
        }
      }
      if (isString(ch)) {
        this.endString = 1;
      } else if (isNumber(ch)) {
        this.endNumber = 1;
      }
    }
    if (this.endString > 0) {
      while (this.startIndex + this.endString < this.buffer.length) {
        const ch = this.buffer.charAt(this.startIndex + this.endString);
        if (ch === '"') {
          return this.endString + 1;
        } else if (ch === '\\') {
          if (this.startIndex + this.endString + 1 < this.buffer.length) {
            this.endString += 2;
          } else {
            break;
          }
        } else {
          this.endString++;
        }
      }
    } else if (this.endNumber > 0) {
      while (this.startIndex + this.endNumber < this.buffer.length) {
        const ch = this.buffer.charAt(this.startIndex + this.endNumber);
        if (jsonNumberCharRe.test(ch)) {
          this.endNumber++;
        } else {
          return this.endNumber;
        }
      }
    }
    return -1;
  }

  removeInitialWhitespace() {
    while (this.startIndex < this.buffer.length) {
      const ch = this.buffer.charAt(this.startIndex);
      if (jsonWhitespaceRe.test(ch)) {
        this.startIndex++;
        this.position++;
      } else {
        break;
      }
    }
    if (this.startIndex >= 8192) {
      this.buffer = this.buffer.substring(this.startIndex);
      this.startIndex = 0;
    }
  }
}

class TokenParsingTransformer {
  state = 'EXPECT_VALUE';
  stackTrueIfObject = [];

  transform(chunk, controller) {
    const { value: inValue, position } = chunk;
    if (this.state === 'DONE') {
      controller.error(new SyntaxError('Unexpected non-whitespace character after JSON at position ' + position));
      return;
    }
    const nextState = this.nextState(this.state, inValue);
    if (nextState === 'ERROR') {
      controller.error(new SyntaxError('Unexpected JSON token at position ' + position));
      return;
    }
    let type;
    let value;
    if (isString(inValue)) {
      try {
        value = JSON.parse(inValue);
      } catch (cause) {
        controller.error(new SyntaxError('Invalid JSON string as position ' + position, { cause }));
        return;
      }
      if (this.state === 'EXPECT_NAME' || this.state === 'EXPECT_NAME_OR_END') type = 'NAME';else
      type = 'STRING';
    } else if (isNumber(inValue)) {
      try {
        value = JSON.parse(inValue);
      } catch (cause) {
        controller.error(new SyntaxError('Invalid JSON number as position ' + position, { cause }));
        return;
      }
      type = 'NUMBER';
    } else if (inValue === 'true') {
      value = true;
      type = 'BOOLEAN';
    } else if (inValue === 'false') {
      value = false;
      type = 'BOOLEAN';
    } else if (inValue === 'null') {
      value = null;
      type = 'NULL';
    } else if (inValue === '{') {
      type = 'BEGIN_OBJECT';
    } else if (inValue === '}') {
      type = 'END_OBJECT';
    } else if (inValue === '[') {
      type = 'BEGIN_ARRAY';
    } else if (inValue === ']') {
      type = 'END_ARRAY';
    }
    if (type) {
      controller.enqueue({ type, value });
    }
    this.state = nextState;
    if (this.state === 'DONE') controller.enqueue({ type: 'END_DOCUMENT' });
  }

  flush(controller) {
    if (this.state !== 'DONE') {
      controller.error(new SyntaxError('Incomplete JSON'));
    }
  }

  nextState(state, value) {
    if (state === 'EXPECT_VALUE' || state === 'EXPECT_VALUE_OR_END') {
      if (value === '{') {
        this.stackTrueIfObject.push(true);
        return 'EXPECT_NAME_OR_END';
      }
      if (value === '[') {
        this.stackTrueIfObject.push(false);
        return 'EXPECT_VALUE_OR_END';
      }
      if (value === 'true' || value === 'false' || value === 'null' || isNumber(value) || isString(value)) {
        if (this.stackTrueIfObject.length === 0) return 'DONE';
        return 'EXPECT_COMMA_OR_END';
      }
      if (state === 'EXPECT_VALUE_OR_END') return this.nextStateEnd(state, value);
    } else if (state === 'EXPECT_NAME') {
      if (isString(value)) return 'EXPECT_COLON';
    } else if (state === 'EXPECT_NAME_OR_END') {
      if (isString(value)) return 'EXPECT_COLON';
      return this.nextStateEnd(state, value);
    } else if (state === 'EXPECT_COMMA_OR_END') {
      if (value === ',') {
        const top = this.stackTrueIfObject[this.stackTrueIfObject.length - 1];
        if (top) return 'EXPECT_NAME';else
        return 'EXPECT_VALUE';
      }
      return this.nextStateEnd(state, value);
    } else if (state === 'EXPECT_COLON') {
      if (value === ':') return 'EXPECT_VALUE';
    }
    return 'ERROR';
  }

  nextStateEnd(state, value) {
    const ending = this.stackTrueIfObject.pop();
    if (ending && value === '}' || !ending && value === ']') {
      if (this.stackTrueIfObject.length === 0) return 'DONE';
      return 'EXPECT_COMMA_OR_END';
    }
    return 'ERROR';
  }
}

/**
 * A streaming JSON reader that allows streaming parsing of JSON data.
 *
 * JsonReader provides methods to navigate through a JSON structure without loading
 * the entire content into memory. This is particularly useful for processing large
 * JSON files or streams efficiently.
 */
class JsonReader {


  closed = false;

  /**
   * Creates a new JsonReader that reads from the specified ReadableStream.
   *
   * @param readable - A ReadableStream containing JSON data as Uint8Array chunks
   */
  constructor(readable) {
    const parsedTokenReadable = readable.
    pipeThrough(new TextDecoderStream()).
    pipeThrough(new TransformStream(new TokenizingTransformer())).
    pipeThrough(new TransformStream(new TokenParsingTransformer()));
    this.reader = parsedTokenReadable.getReader();
  }

  /**
   * Closes the reader and releases any resources associated with it.
   *
   * @returns A Promise that resolves when the reader has been closed
   */
  async close() {
    if (this.closed) return;
    this.closed = true;
    await this.reader.cancel();
  }

  async peekNextEvent() {
    if (this.closed) throw new Error('JsonReader is closed');
    if (this.nextEvent) return this.nextEvent;
    const { value, done } = await this.reader.read();
    if (done) throw new SyntaxError('Incomplete JSON');
    this.nextEvent = value;
    if (value.type === "END_DOCUMENT") {
      // ensure no additional tokens
      await this.reader.read();
    }
    return this.nextEvent;
  }

  /**
   * Looks at the next token in the JSON stream without consuming it.
   *
   * This method allows you to check what type of token is coming next
   * without advancing the reader position.
   *
   * @returns A Promise that resolves to the type of the next token
   * @throws Error if the reader is closed or the JSON is incomplete
   */
  async peek() {
    return (await this.peekNextEvent()).type;
  }

  async expectAndConsume(type) {
    const nextEvent = await this.peekNextEvent();
    if (nextEvent.type !== type) {
      throw new Error(`Not at ${type}, at ${nextEvent.type}`);
    }
    const res = nextEvent.value;
    this.nextEvent = undefined;
    return res;
  }

  /**
   * Consumes the beginning of a JSON array.
   *
   * @throws Error if the next token is not the beginning of an array ('[')
   */
  async beginArray() {
    await this.expectAndConsume('BEGIN_ARRAY');
  }

  /**
   * Consumes the beginning of a JSON object.
   *
   * @throws Error if the next token is not the beginning of an object ('{')
   */
  async beginObject() {
    await this.expectAndConsume('BEGIN_OBJECT');
  }

  /**
   * Consumes the end of a JSON array.
   *
   * @throws Error if the next token is not the end of an array (']')
   */
  async endArray() {
    await this.expectAndConsume('END_ARRAY');
  }

  /**
   * Consumes the end of a JSON object.
   *
   * @throws Error if the next token is not the end of an object ('}')
   */
  async endObject() {
    await this.expectAndConsume('END_OBJECT');
  }

  /**
   * Checks if there are more elements in the current array or object.
   *
   * This method is typically used in a while loop to iterate through
   * all elements in an array or all properties in an object.
   *
   * @returns A Promise that resolves to true if there are more elements,
   *          or false if the end of the current array or object has been reached
   * @throws Error if the reader is closed or the JSON is incomplete
   *
   * @example
   * // Iterate through an array
   * await reader.beginArray();
   * while (await reader.hasNext()) {
   *   const value = await reader.nextString();
   *   console.log(value);
   * }
   * await reader.endArray();
   */
  async hasNext() {
    const nextEvent = await this.peekNextEvent();
    if (nextEvent.type === 'END_DOCUMENT' || nextEvent.type === 'END_ARRAY' || nextEvent.type === 'END_OBJECT') {
      return false;
    }
    return true;
  }

  /**
   * Reads the name of the next property in a JSON object.
   *
   * @returns A Promise that resolves to the name of the property
   * @throws Error if the next token is not a property name
   *
   * @example
   * await reader.beginObject();
   * while (await reader.hasNext()) {
   *   const propertyName = await reader.nextName();
   *   // Process the property based on its name
   * }
   * await reader.endObject();
   */
  async nextName() {
    return this.expectAndConsume('NAME');
  }

  /**
   * Reads the next JSON primitive value.
   *
   * @returns A Promise that resolves to the primitive value
   * @throws Error if the next token is not a primitive value
   */
  async nextJsonPrimitive() {
    const nextEvent = await this.peekNextEvent();
    if (nextEvent.type === 'STRING' || nextEvent.type === 'NUMBER' || nextEvent.type === 'BOOLEAN' || nextEvent.type === 'NULL') {
      const res = nextEvent.value;
      this.nextEvent = undefined;
      return res;
    }
    throw new Error(`Not at JSON primitive, at ${nextEvent.type}`);
  }

  /**
   * Reads the next value as a string.
   *
   * If the next value is already a string, it is returned as is.
   * If the next value is another primitive type, it is converted
   * to a string using JSON.stringify.
   *
   * @returns A Promise that resolves to the string value
   * @throws Error if the next token is not a primitive value
   */
  async nextString() {
    const s = await this.nextJsonPrimitive();
    if (typeof s === 'string') return s;
    return JSON.stringify(s);
  }

  async nextPrimitiveOfType(type) {
    const nextEvent = await this.peekNextEvent();
    if (nextEvent.type === type) {
      const res = nextEvent.value;
      this.nextEvent = undefined;
      return res;
    }
    throw new Error(`Not at ${type}, at ${nextEvent.type}`);
  }

  /**
   * Reads the next value as a number.
   *
   * @returns A Promise that resolves to the number value
   * @throws Error if the next token is not a number
   */
  async nextNumber() {
    return this.nextPrimitiveOfType("NUMBER");
  }

  /**
   * Reads the next value as a boolean.
   *
   * @returns A Promise that resolves to the boolean value
   * @throws Error if the next token is not a boolean
   */
  async nextBoolean() {
    return this.nextPrimitiveOfType("BOOLEAN");
  }

  /**
   * Reads the next value as null.
   *
   * @returns A Promise that resolves to null
   * @throws Error if the next token is not null
   */
  async nextNull() {
    return this.nextPrimitiveOfType("NULL");
  }

  /**
   * Skips the next value in the JSON stream.
   *
   * @throws Error if there is no more JSON to read, or if the next token is a property name
   *
   * @example
   * // Skip properties you don't care about
   * await reader.beginObject();
   * while (await reader.hasNext()) {
   *   const name = await reader.nextName();
   *   if (name === "importantProperty") {
   *     const value = await reader.nextString();
   *     console.log(value);
   *   } else {
   *     await reader.skipValue(); // Skip this property
   *   }
   * }
   * await reader.endObject();
   */
  async skipValue() {
    let nextEvent = await this.peekNextEvent();
    if (nextEvent.type === "END_DOCUMENT") throw new Error('No more JSON');
    if (nextEvent.type === "NAME") throw new Error('At NAME, not able to skipValue');
    let count = 0;
    while (true) {
      if (nextEvent.type === 'BEGIN_ARRAY' || nextEvent.type === 'BEGIN_OBJECT') count++;
      if (nextEvent.type === 'END_ARRAY' || nextEvent.type === 'END_OBJECT') count--;
      this.nextEvent = undefined;
      if (count === 0) break;
      nextEvent = await this.peekNextEvent();
    }
  }

  /**
   * Reads the next complete JSON value (object, array, or primitive).
   *
   * This method parses and returns the entire next value in the JSON stream,
   * regardless of its complexity. It's useful when you want to read a complete
   * JSON structure without manually navigating through it.
   *
   * @returns A Promise that resolves to the parsed JSON value
   * @throws Error if the JSON is invalid or incomplete
   *
   * @example
   * // Read a complete JSON object
   * const result = await reader.nextJson();
   * console.log(result.content.name);
   */
  async nextJson() {
    const stack = [];
    while (true) {
      const { type: nextToken, value } = await this.peekNextEvent();
      let json;
      if (nextToken === "BEGIN_ARRAY") {
        stack.push({ value: [] });
      } else if (nextToken === "BEGIN_OBJECT") {
        stack.push({ value: {} });
      } else if (nextToken === "END_ARRAY" || nextToken === "END_OBJECT") {
        json = stack.pop().value;
      } else if (nextToken === "NAME") {
        const top = stack[stack.length - 1];
        top.name = value;
      } else {
        json = value;
      }
      this.nextEvent = undefined;
      if (json !== undefined) {
        if (stack.length === 0) {
          // ensure that garbage at end of document results in error
          await this.peekNextEvent();
          return json;
        }
        const top = stack[stack.length - 1];
        if (top.name !== undefined) {
          const obj = top.value;
          // Property named __proto__ requires special handling
          if (top.name !== '__proto__') obj[top.name] = json;else
          top.value = { ...obj, ['__proto__']: json };
        } else top.value.push(json);
        top.name = undefined;
      }
    }
  }
}

/* eslint-disable @typescript-eslint/no-explicit-any */


/**
 * Parses JSON data from various input sources using a streaming approach.
 *
 * This function provides a way to parse JSON content from different input types without
 * loading the entire content into memory at once. It uses the {@link JsonReader} class
 * internally to perform the streaming parsing.
 *
 * This function is mostly used for testing.
 *
 * @param input - The JSON data to parse
 *
 * @returns A Promise that resolves to the parsed JSON value
 */
async function jsonParse(input) {
  let readable;
  if (input instanceof ReadableStream) {
    readable = input;
  } else {
    readable = new Response(new Blob([input])).body;
  }
  const jsonReader = new JsonReader(readable);
  try {
    return await jsonReader.nextJson();
  } finally {
    await jsonReader.close();
  }
}

export { JsonReader, jsonParse };
//# sourceMappingURL=json-streaming-1.0.0.js.map
