# @cnri/json-streaming

This library provides a streaming API for reading JSON. This can be useful when needing to read from very large JSON objects. Instead of reading the entire object into memory, objects can be read off of it one by one. This library can be used in both browser and Node.js environments.

## Usage

The library provides two main ways to parse JSON:

1. `JsonReader` - A streaming reader for incremental JSON parsing
2. `jsonParse` - A simple function for parsing entire JSON content using JsonReader

### Using JsonReader

The `JsonReader` class provides a high level of control over the parsing process, allowing you to read JSON incrementally:

```javascript
import { JsonReader } from '@cnri/json-streaming';

// Create a reader from a fetch response
const response = await fetch('https://example.com/large-data.json');
const reader = new JsonReader(response.body);

try {
  // Read a JSON object incrementally
  await reader.beginObject();

  while (await reader.hasNext()) {
    const propertyName = await reader.nextName();

    if (propertyName === 'results') {
      // Process an array
      await reader.beginArray();

      while (await reader.hasNext()) {
        // Process each result individually
        const result = await reader.nextJson();
        console.log(result.id);
      }

      await reader.endArray();
    } else if (propertyName === 'size') {
      // Read a number
      const count = await reader.nextNumber();
      console.log(`Total results: ${count}`);
    } else {
      // Skip properties we don't care about
      console.log(`Skipping property: ${propertyName}`);
      await reader.skipValue();
    }
  }

  await reader.endObject();
} finally {
  // Always close the reader when done
  await reader.close();
}
```

### Using jsonParse

The `jsonParse` function provides a way to parse JSON from various input sources. Notably, when parsing a ReadableStream, jsonParse should be more memory-efficient than reading the entire ReadableStream into a string and then using JSON.parse.

```javascript
import { jsonParse } from '@cnri/json-streaming';

// Parse JSON from a string
const data = await jsonParse('{"item": "Widget", "count": 1000}');
console.log(data.item); // "Widget"

// Parse JSON from a fetch response
const response = await fetch('https://example.com/data.json');
const result = await jsonParse(response.body);
console.log(result);
```
