/**
 * Provides functions for encoding and decoding data in Base64 format.
 * Supports both standard Base64 and URL-safe Base64.
 * @namespace
 */
const Base64 = { decode: decode$1, encode: encode$1, encodeUrlSafe };

const un$1 = undefined;

const base64DecodeArray = [
un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1,
un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1,
un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, 62, un$1, 62, un$1, 63,
52, 53, 54, 55, 56, 57, 58, 59, 60, 61, un$1, un$1, un$1, un$1, un$1, un$1,
// eslint-disable-next-line @stylistic/no-multi-spaces
un$1, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14,
15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, un$1, un$1, un$1, un$1, 63,
un$1, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51];


const base64EncodeArray = [
'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P',
'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f',
'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v',
'w', 'x', 'y', 'z', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '+', '/'];


const base64UrlEncodeArray = [
'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P',
'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f',
'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v',
'w', 'x', 'y', 'z', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '-', '_'];


/**
 * Decodes a Base64 string into a byte array.
 *
 * @param s - The Base64 string to decode
 * @returns A Uint8Array containing the decoded bytes
 */
function decode$1(s) {
  const res = new Uint8Array(calcNumBytes$3(s));
  let pos = 0;
  let inFour = 0;
  let accum = 0;
  for (let i = 0; i < s.length; i++) {
    const code = base64DecodeArray[s.charCodeAt(i)];
    if (code === undefined) continue;
    if (inFour === 0) {
      accum = code << 2;
    } else if (inFour === 1) {
      accum |= code >>> 4;
      res[pos++] = accum;
      accum = (code & 0x0F) << 4;
    } else if (inFour === 2) {
      accum |= code >>> 2;
      res[pos++] = accum;
      accum = (code & 0x03) << 6;
    } else {
      accum |= code;
      res[pos++] = accum;
    }
    inFour = (inFour + 1) % 4;
  }
  return res.subarray(0, pos);
}

function calcNumBytes$3(str) {
  let len = str.length;
  // eslint-disable-next-line @typescript-eslint/prefer-string-starts-ends-with
  if (str[str.length - 1] === '=') {
    len -= 1;
    if (str[str.length - 2] === '=') len -= 1;
  }
  const mod = len % 4;
  if (mod === 0) return 3 * len / 4;
  if (mod === 1) return 3 * (len - 1) / 4;
  if (mod === 2) return 3 * (len - 2) / 4 + 1;
  return 3 * (len - 3) / 4 + 2;
}

function genericBase64EncoderFunction(arr, thisBase64EncodeArray, usePad) {
  let s = '';
  let accum = 0;
  let inThree = 0;
  for (const code of arr) {
    if (inThree === 0) {
      s += thisBase64EncodeArray[code >>> 2];
      accum = (code & 0x03) << 4;
    } else if (inThree === 1) {
      accum |= code >>> 4;
      s += thisBase64EncodeArray[accum];
      accum = (code & 0x0F) << 2;
    } else {
      accum |= code >>> 6;
      s += thisBase64EncodeArray[accum];
      s += thisBase64EncodeArray[code & 0x3F];
    }
    inThree = (inThree + 1) % 3;
  }
  if (inThree > 0) {
    s += thisBase64EncodeArray[accum];
    if (usePad) {
      const pad = '=';
      s += pad;
      if (inThree === 1) s += pad;
    }
  }
  return s;
}

/**
 * Encodes a byte array into a standard Base64 string, with padding.
 *
 * @param arr - The byte array to encode
 * @returns The Base64 encoded string
 */
function encode$1(arr) {
  return genericBase64EncoderFunction(arr, base64EncodeArray, true);
}

/**
 * Encodes a byte array into a URL-safe Base64 string.
 *
 * @param arr - The byte array to encode
 * @returns The URL-safe Base64 encoded string
 */
function encodeUrlSafe(arr) {
  return genericBase64EncoderFunction(arr, base64UrlEncodeArray, false);
}

/**
 * Provides functions for encoding and decoding data in hexadecimal format.
 * @namespace
 */
const Hex = { decode, encode };

const un = undefined;

const hexDecodeArray = [
un, un, un, un, un, un, un, un, un, un, un, un, un, un, un, un,
un, un, un, un, un, un, un, un, un, un, un, un, un, un, un, un,
un, un, un, un, un, un, un, un, un, un, un, un, un, un, un, un,
// eslint-disable-next-line @stylistic/no-multi-spaces
0, 1, 2, 3, 4, 5, 6, 7, 8, 9, un, un, un, un, un, un,
un, 10, 11, 12, 13, 14, 15, 16, un, un, un, un, un, un, un, un,
un, un, un, un, un, un, un, un, un, un, un, un, un, un, un, un,
un, 10, 11, 12, 13, 14, 15, 16];


const hexEncodeArray = [
'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'];


/**
 * Decodes a hexadecimal string into a byte array.
 *
 * @param str - The hexadecimal string to decode
 * @returns A Uint8Array containing the decoded bytes
 */
function decode(str) {
  const res = new Uint8Array(calcNumBytes$2(str));
  let pos = 0;
  let inTwo = 0;
  let accum = 0;
  for (let i = 0; i < str.length; i++) {
    const code = hexDecodeArray[str.charCodeAt(i)];
    if (code === undefined) continue;
    if (inTwo === 0) {
      accum = code << 4;
    } else {
      accum |= code;
      res[pos++] = accum;
    }
    inTwo = (inTwo + 1) % 2;
  }
  return res.subarray(0, pos);
}

function calcNumBytes$2(str) {
  return Math.floor(str.length / 2);
}

/**
 * Encodes a byte array into a hexadecimal string.
 * Uses uppercase letters (A-F) for values 10-15.
 *
 * @param arr - The byte array to encode
 * @returns The hexadecimal encoded string (uppercase)
 */
function encode(arr) {
  let s = '';
  for (const code of arr) {
    s += hexEncodeArray[code >>> 4];
    s += hexEncodeArray[code & 0x0F];
  }
  return s;
}

/**
 * Provides functions for converting between strings and UTF-8 encoded byte arrays.
 * Also includes utility functions to detect binary data.
 * @namespace
 */
const Utf8 = { toBytes: toBytes$1, toString: toString$1, looksLikeBinary, stringLooksLikeBinary };

/**
 * Converts a string to a UTF-8 encoded byte array.
 *
 * @param str - The string to convert
 * @returns A Uint8Array containing the UTF-8 encoded bytes
 */
function toBytes$1(str) {
  const res = new Uint8Array(calcNumBytes$1(str));
  let pos = 0;
  for (let i = 0; i < str.length; i++) {
    let code = str.charCodeAt(i);
    if (code <= 0x7F) {
      res[pos++] = code;
    } else if (code <= 0x7FF) {
      res[pos++] = 0xC0 | code >>> 6;
      res[pos++] = 0x80 | code & 0x3F;
    } else if (0xD800 <= code && code <= 0xDBFF) {
      if (i + 1 < str.length) {
        const next = str.charCodeAt(i + 1);
        if (0xDC00 <= next && next <= 0xDFFF) {
          i++;
          code = ((code - 0xD800) * 0x400 | next - 0xDC00) + 0x10000;
          res[pos++] = 0xF0 | code >>> 18;
          res[pos++] = 0x80 | code >>> 12 & 0x3F;
          res[pos++] = 0x80 | code >>> 6 & 0x3F;
          res[pos++] = 0x80 | code & 0x3F;
          continue;
        }
      }
      // bare surrogate
      res[pos++] = 0xE0 | code >>> 12;
      res[pos++] = 0x80 | code >>> 6 & 0x3F;
      res[pos++] = 0x80 | code & 0x3F;
    } else {
      res[pos++] = 0xE0 | code >>> 12;
      res[pos++] = 0x80 | code >>> 6 & 0x3F;
      res[pos++] = 0x80 | code & 0x3F;
    }
  }
  return res;
}

function calcNumBytes$1(str) {
  let res = 0;
  for (let i = 0; i < str.length; i++) {
    const code = str.charCodeAt(i);
    if (code <= 0x7F) res += 1;else
    if (code <= 0x7FF) res += 2;else
    if (0xD800 <= code && code <= 0xDBFF) {
      if (i + 1 < str.length) {
        const next = str.charCodeAt(i + 1);
        if (0xDC00 <= next && next <= 0xDFFF) {
          i++;
          res += 4;
          continue;
        }
      }
      // bare surrogate
      res += 3;
    } else res += 3;
  }
  return res;
}

/**
 * Converts a UTF-8 encoded byte array to a string.
 *
 * @param arr - The UTF-8 encoded byte array
 * @returns The decoded string
 */
function toString$1(arr) {
  let str = '';
  for (let i = 0; i < arr.length; i++) {
    let code = arr[i];
    let thisValid = false;
    if (code <= 0x7F) {
      thisValid = true;
      //            if(!binary && (code <= 0x08 || (0x0E <= code && code < 0x20) || code===0x7F)) binary = true;
      str += String.fromCharCode(code);
    } else if (code <= 0xC1 || code >= 0xF5) {
      thisValid = false;
    } else if (code <= 0xDF) {
      if (i + 1 < arr.length) {
        const c2 = arr[i + 1];
        if (0x80 <= c2 && c2 <= 0xBF) {
          thisValid = true;
          i++;
          str += String.fromCharCode((code & 0x1F) << 6 | c2 & 0x3F);
        }
      }
    } else if (code <= 0xEF) {
      if (i + 2 < arr.length) {
        const c2 = arr[i + 1];
        const c3 = arr[i + 2];
        if (0x80 <= c2 && c2 <= 0xBF && 0x80 <= c3 && c3 <= 0xBF && !(code === 0xE0 && c2 <= 0x9F)) {
          thisValid = true;
          i += 2;
          str += String.fromCharCode((code & 0x0F) << 12 | (c2 & 0x3F) << 6 | c3 & 0x3F);
        }
      }
    } else {
      // eslint-disable-next-line no-lonely-if
      if (i + 3 < arr.length) {
        const c2 = arr[i + 1];
        const c3 = arr[i + 2];
        const c4 = arr[i + 3];
        if (0x80 <= c2 && c2 <= 0xBF && 0x80 <= c3 && c3 <= 0xBF && 0x80 <= c4 && c4 <= 0xBF && !(code === 0xF0 && c2 <= 0x8F)) {
          code = (code & 0x07) << 18 | (c2 & 0x3F) << 12 | (c3 & 0x3F) << 6 | c4 & 0x3F;
          if (code <= 0x10FFFF) {
            thisValid = true;
            i += 3;
            code -= 0x10000;
            str += String.fromCharCode(0xD800 + (code >> 10), 0xDC00 + (code & 0x3FF));
          }
        }
      }
    }
    if (!thisValid) {
      str += String.fromCharCode(0xFFFD);
    }
  }
  return str;
}

/**
 * Checks if a byte array likely contains binary data rather than valid UTF-8 text.
 *
 * @param arr - The byte array to check
 * @returns True if the array likely contains binary data, false otherwise
 */
function looksLikeBinary(arr) {
  for (let i = 0; i < arr.length; i++) {
    const code = arr[i];
    if (code <= 0x7F) {
      if (code <= 0x08 || 0x0E <= code && code < 0x20 || code === 0x7F) return true;
    } else if (code <= 0xC1 || code >= 0xF5) {
      return true;
    } else if (code <= 0xDF) {
      if (i + 1 >= arr.length) return true;
      const c2 = arr[++i];
      if (!(0x80 <= c2 && c2 <= 0xBF)) return true;
    } else if (code <= 0xEF) {
      if (i + 2 >= arr.length) return true;
      const c2 = arr[++i];
      const c3 = arr[++i];
      if (!(0x80 <= c2 && c2 <= 0xBF && 0x80 <= c3 && c3 <= 0xBF && !(code === 0xE0 && c2 <= 0x9F))) return true;
    } else {
      if (i + 3 >= arr.length) return true;
      const c2 = arr[++i];
      const c3 = arr[++i];
      const c4 = arr[++i];
      if (!(0x80 <= c2 && c2 <= 0xBF && 0x80 <= c3 && c3 <= 0xBF && 0x80 <= c4 && c4 <= 0xBF && !(code === 0xF0 && c2 <= 0x8F))) return true;
    }
  }
  return false;
}

/**
 * Checks if a string likely contains binary data.
 *
 * @param str - The string to check
 * @returns True if the string likely contains binary data, false otherwise
 */
function stringLooksLikeBinary(str) {
  for (let i = 0; i < str.length; i++) {
    const code = str.charCodeAt(i);
    if (code <= 0x08 || 0x0E <= code && code < 0x20 || code === 0x7F || code === 0xFFFD) return true;
  }
  return false;
}

/**
 * Provides functions for converting between strings and UTF-16 encoded byte arrays.
 * @namespace
 */
const Utf16 = { toBytes, toString };

/**
 * Converts a string to a UTF-16 encoded byte array.
 *
 * @param str - The string to convert
 * @returns A Uint8Array containing the UTF-16 encoded bytes
 */
function toBytes(str) {
  const arr = new Uint8Array(calcNumBytes(str));
  let pos = 0;
  for (let i = 0; i < str.length; i++) {
    const code = str.charCodeAt(i);
    arr[pos++] = code >>> 8;
    arr[pos++] = code & 0xFF;
  }
  return arr;
}

function calcNumBytes(str) {
  return str.length * 2;
}

/**
 * Converts a UTF-16 encoded byte array to a string.
 *
 * @param arr - The UTF-16 encoded byte array
 * @returns The decoded string
 */
function toString(arr) {
  let res = '';
  let i;
  for (i = 0; i + 1 < arr.length; i += 2) {
    const code = arr[i] << 8 | arr[i + 1];
    res += String.fromCharCode(code);
  }
  if (i < arr.length) res += String.fromCharCode(0xFFFD);
  return res;
}

export { Base64, Hex, Utf16, Utf8 };
//# sourceMappingURL=encoding-1.0.0.js.map
