const jsonPunctRe = /[[\]{},:]/;
const jsonNumberCharRe = /[-+eE0-9.]/;
const jsonWhitespaceRe = /[ \t\r\n]/;

const literals = ['true', 'false', 'null'];
const partialLiteralRe = /^(?:t(?:ru?)?|f(?:a(?:ls?)?)?|n(?:ul?)?)$/; // t|tr|tru|f|fa|fal|fals|n|nu|nul

const YIELD_LIMIT = 2000; // yield roughly every this many characters; without this there are performance issues on large inputs
// The number 2000 was obtained experimentally -- 1000 and 2000 and 4000 are roughly the same; 500 is much slower; 16000 is much slower

function isNumber(value) {
  return !!value && jsonNumberCharRe.test(value.charAt(0));
}

function isString(value) {
  return value.startsWith('"');
}






class TokenizingTransformer {
  position = 0; // the position of startIndex in the entire stream
  startIndex = 0; // the index within the current buffer up to which has been tokenized
  buffer = '';
  endString = 0;
  endNumber = 0;

  async transform(chunk, controller) {
    for (let i = 0; i < chunk.length; i += YIELD_LIMIT) {
      const bit = chunk.substring(i, i + YIELD_LIMIT);
      this.buffer += bit;
      this.tokenize(controller, false);
      if (i + YIELD_LIMIT < chunk.length) await new Promise((r) => setTimeout(r, 0));
    }
  }

  flush(controller) {
    this.tokenize(controller, true);
  }

  enqueue(controller, size) {
    const value = this.buffer.substring(this.startIndex, this.startIndex + size);
    controller.enqueue({ value, position: this.position });
    this.startIndex += size;
    if (this.startIndex >= 8192) {
      this.buffer = this.buffer.substring(this.startIndex);
      this.startIndex = 0;
    }
    this.position += size;
    this.endString = 0;
    this.endNumber = 0;
  }

  tokenize(controller, final) {
    while (this.buffer.length > this.startIndex) {
      if (this.endString === 0 && this.endNumber === 0) {
        this.removeInitialWhitespace();
        if (this.buffer.length === this.startIndex) break;
      }
      const tokenSize = this.findOneToken();
      if (tokenSize > 0) {
        this.enqueue(controller, tokenSize);
        continue;
      } else if (this.endString > 0) {
        if (!final) break;
        controller.error(new SyntaxError('Unterminated string in JSON at position ' + (this.position + this.endString)));
        return;
      } else if (this.endNumber > 0) {
        if (!final) break;
        this.enqueue(controller, this.endNumber);
      } else {
        if (!final) {
          if (this.buffer.length - this.startIndex <= 4) {
            if (partialLiteralRe.test(this.buffer.substring(this.startIndex))) break;
          }
        }
        // garbage
        controller.error(new SyntaxError('Invalid JSON token at position ' + this.position));
        return;
      }
    }
  }

  findOneToken() {
    if (this.endString === 0 && this.endNumber === 0) {
      const ch = this.buffer.charAt(this.startIndex);
      if (jsonPunctRe.test(ch)) {
        return 1;
      }
      for (const literal of literals) {
        if (this.buffer.startsWith(literal, this.startIndex)) {
          return literal.length;
        }
      }
      if (isString(ch)) {
        this.endString = 1;
      } else if (isNumber(ch)) {
        this.endNumber = 1;
      }
    }
    if (this.endString > 0) {
      while (this.startIndex + this.endString < this.buffer.length) {
        const ch = this.buffer.charAt(this.startIndex + this.endString);
        if (ch === '"') {
          return this.endString + 1;
        } else if (ch === '\\') {
          if (this.startIndex + this.endString + 1 < this.buffer.length) {
            this.endString += 2;
          } else {
            break;
          }
        } else {
          this.endString++;
        }
      }
    } else if (this.endNumber > 0) {
      while (this.startIndex + this.endNumber < this.buffer.length) {
        const ch = this.buffer.charAt(this.startIndex + this.endNumber);
        if (jsonNumberCharRe.test(ch)) {
          this.endNumber++;
        } else {
          return this.endNumber;
        }
      }
    }
    return -1;
  }

  removeInitialWhitespace() {
    while (this.startIndex < this.buffer.length) {
      const ch = this.buffer.charAt(this.startIndex);
      if (jsonWhitespaceRe.test(ch)) {
        this.startIndex++;
        this.position++;
      } else {
        break;
      }
    }
    if (this.startIndex >= 8192) {
      this.buffer = this.buffer.substring(this.startIndex);
      this.startIndex = 0;
    }
  }
}

class TokenParsingTransformer {
  state = 'EXPECT_VALUE';
  stackTrueIfObject = [];

  transform(chunk, controller) {
    const { value: inValue, position } = chunk;
    if (this.state === 'DONE') {
      controller.error(new SyntaxError('Unexpected non-whitespace character after JSON at position ' + position));
      return;
    }
    const nextState = this.nextState(this.state, inValue);
    if (nextState === 'ERROR') {
      controller.error(new SyntaxError('Unexpected JSON token at position ' + position));
      return;
    }
    let type;
    let value;
    if (isString(inValue)) {
      try {
        value = JSON.parse(inValue);
      } catch (cause) {
        controller.error(new SyntaxError('Invalid JSON string as position ' + position, { cause }));
        return;
      }
      if (this.state === 'EXPECT_NAME' || this.state === 'EXPECT_NAME_OR_END') type = 'NAME';else
      type = 'STRING';
    } else if (isNumber(inValue)) {
      try {
        value = JSON.parse(inValue);
      } catch (cause) {
        controller.error(new SyntaxError('Invalid JSON number as position ' + position, { cause }));
        return;
      }
      type = 'NUMBER';
    } else if (inValue === 'true') {
      value = true;
      type = 'BOOLEAN';
    } else if (inValue === 'false') {
      value = false;
      type = 'BOOLEAN';
    } else if (inValue === 'null') {
      value = null;
      type = 'NULL';
    } else if (inValue === '{') {
      type = 'BEGIN_OBJECT';
    } else if (inValue === '}') {
      type = 'END_OBJECT';
    } else if (inValue === '[') {
      type = 'BEGIN_ARRAY';
    } else if (inValue === ']') {
      type = 'END_ARRAY';
    }
    if (type) {
      controller.enqueue({ type, value });
    }
    this.state = nextState;
    if (this.state === 'DONE') controller.enqueue({ type: 'END_DOCUMENT' });
  }

  flush(controller) {
    if (this.state !== 'DONE') {
      controller.error(new SyntaxError('Incomplete JSON'));
    }
  }

  nextState(state, value) {
    if (state === 'EXPECT_VALUE' || state === 'EXPECT_VALUE_OR_END') {
      if (value === '{') {
        this.stackTrueIfObject.push(true);
        return 'EXPECT_NAME_OR_END';
      }
      if (value === '[') {
        this.stackTrueIfObject.push(false);
        return 'EXPECT_VALUE_OR_END';
      }
      if (value === 'true' || value === 'false' || value === 'null' || isNumber(value) || isString(value)) {
        if (this.stackTrueIfObject.length === 0) return 'DONE';
        return 'EXPECT_COMMA_OR_END';
      }
      if (state === 'EXPECT_VALUE_OR_END') return this.nextStateEnd(state, value);
    } else if (state === 'EXPECT_NAME') {
      if (isString(value)) return 'EXPECT_COLON';
    } else if (state === 'EXPECT_NAME_OR_END') {
      if (isString(value)) return 'EXPECT_COLON';
      return this.nextStateEnd(state, value);
    } else if (state === 'EXPECT_COMMA_OR_END') {
      if (value === ',') {
        const top = this.stackTrueIfObject[this.stackTrueIfObject.length - 1];
        if (top) return 'EXPECT_NAME';else
        return 'EXPECT_VALUE';
      }
      return this.nextStateEnd(state, value);
    } else if (state === 'EXPECT_COLON') {
      if (value === ':') return 'EXPECT_VALUE';
    }
    return 'ERROR';
  }

  nextStateEnd(state, value) {
    const ending = this.stackTrueIfObject.pop();
    if (ending && value === '}' || !ending && value === ']') {
      if (this.stackTrueIfObject.length === 0) return 'DONE';
      return 'EXPECT_COMMA_OR_END';
    }
    return 'ERROR';
  }
}

/**
 * A streaming JSON reader that allows streaming parsing of JSON data.
 *
 * JsonReader provides methods to navigate through a JSON structure without loading
 * the entire content into memory. This is particularly useful for processing large
 * JSON files or streams efficiently.
 */
class JsonReader {


  closed = false;

  /**
   * Creates a new JsonReader that reads from the specified ReadableStream.
   *
   * @param readable - A ReadableStream containing JSON data as Uint8Array chunks
   */
  constructor(readable) {
    const parsedTokenReadable = readable.
    pipeThrough(new TextDecoderStream()).
    pipeThrough(new TransformStream(new TokenizingTransformer())).
    pipeThrough(new TransformStream(new TokenParsingTransformer()));
    this.reader = parsedTokenReadable.getReader();
  }

  /**
   * Closes the reader and releases any resources associated with it.
   *
   * @returns A Promise that resolves when the reader has been closed
   */
  async close() {
    if (this.closed) return;
    this.closed = true;
    await this.reader.cancel();
  }

  async peekNextEvent() {
    if (this.closed) throw new Error('JsonReader is closed');
    if (this.nextEvent) return this.nextEvent;
    const { value, done } = await this.reader.read();
    if (done) throw new SyntaxError('Incomplete JSON');
    this.nextEvent = value;
    if (value.type === "END_DOCUMENT") {
      // ensure no additional tokens
      await this.reader.read();
    }
    return this.nextEvent;
  }

  /**
   * Looks at the next token in the JSON stream without consuming it.
   *
   * This method allows you to check what type of token is coming next
   * without advancing the reader position.
   *
   * @returns A Promise that resolves to the type of the next token
   * @throws Error if the reader is closed or the JSON is incomplete
   */
  async peek() {
    return (await this.peekNextEvent()).type;
  }

  async expectAndConsume(type) {
    const nextEvent = await this.peekNextEvent();
    if (nextEvent.type !== type) {
      throw new Error(`Not at ${type}, at ${nextEvent.type}`);
    }
    const res = nextEvent.value;
    this.nextEvent = undefined;
    return res;
  }

  /**
   * Consumes the beginning of a JSON array.
   *
   * @throws Error if the next token is not the beginning of an array ('[')
   */
  async beginArray() {
    await this.expectAndConsume('BEGIN_ARRAY');
  }

  /**
   * Consumes the beginning of a JSON object.
   *
   * @throws Error if the next token is not the beginning of an object ('{')
   */
  async beginObject() {
    await this.expectAndConsume('BEGIN_OBJECT');
  }

  /**
   * Consumes the end of a JSON array.
   *
   * @throws Error if the next token is not the end of an array (']')
   */
  async endArray() {
    await this.expectAndConsume('END_ARRAY');
  }

  /**
   * Consumes the end of a JSON object.
   *
   * @throws Error if the next token is not the end of an object ('}')
   */
  async endObject() {
    await this.expectAndConsume('END_OBJECT');
  }

  /**
   * Checks if there are more elements in the current array or object.
   *
   * This method is typically used in a while loop to iterate through
   * all elements in an array or all properties in an object.
   *
   * @returns A Promise that resolves to true if there are more elements,
   *          or false if the end of the current array or object has been reached
   * @throws Error if the reader is closed or the JSON is incomplete
   *
   * @example
   * // Iterate through an array
   * await reader.beginArray();
   * while (await reader.hasNext()) {
   *   const value = await reader.nextString();
   *   console.log(value);
   * }
   * await reader.endArray();
   */
  async hasNext() {
    const nextEvent = await this.peekNextEvent();
    if (nextEvent.type === 'END_DOCUMENT' || nextEvent.type === 'END_ARRAY' || nextEvent.type === 'END_OBJECT') {
      return false;
    }
    return true;
  }

  /**
   * Reads the name of the next property in a JSON object.
   *
   * @returns A Promise that resolves to the name of the property
   * @throws Error if the next token is not a property name
   *
   * @example
   * await reader.beginObject();
   * while (await reader.hasNext()) {
   *   const propertyName = await reader.nextName();
   *   // Process the property based on its name
   * }
   * await reader.endObject();
   */
  async nextName() {
    return this.expectAndConsume('NAME');
  }

  /**
   * Reads the next JSON primitive value.
   *
   * @returns A Promise that resolves to the primitive value
   * @throws Error if the next token is not a primitive value
   */
  async nextJsonPrimitive() {
    const nextEvent = await this.peekNextEvent();
    if (nextEvent.type === 'STRING' || nextEvent.type === 'NUMBER' || nextEvent.type === 'BOOLEAN' || nextEvent.type === 'NULL') {
      const res = nextEvent.value;
      this.nextEvent = undefined;
      return res;
    }
    throw new Error(`Not at JSON primitive, at ${nextEvent.type}`);
  }

  /**
   * Reads the next value as a string.
   *
   * If the next value is already a string, it is returned as is.
   * If the next value is another primitive type, it is converted
   * to a string using JSON.stringify.
   *
   * @returns A Promise that resolves to the string value
   * @throws Error if the next token is not a primitive value
   */
  async nextString() {
    const s = await this.nextJsonPrimitive();
    if (typeof s === 'string') return s;
    return JSON.stringify(s);
  }

  async nextPrimitiveOfType(type) {
    const nextEvent = await this.peekNextEvent();
    if (nextEvent.type === type) {
      const res = nextEvent.value;
      this.nextEvent = undefined;
      return res;
    }
    throw new Error(`Not at ${type}, at ${nextEvent.type}`);
  }

  /**
   * Reads the next value as a number.
   *
   * @returns A Promise that resolves to the number value
   * @throws Error if the next token is not a number
   */
  async nextNumber() {
    return this.nextPrimitiveOfType("NUMBER");
  }

  /**
   * Reads the next value as a boolean.
   *
   * @returns A Promise that resolves to the boolean value
   * @throws Error if the next token is not a boolean
   */
  async nextBoolean() {
    return this.nextPrimitiveOfType("BOOLEAN");
  }

  /**
   * Reads the next value as null.
   *
   * @returns A Promise that resolves to null
   * @throws Error if the next token is not null
   */
  async nextNull() {
    return this.nextPrimitiveOfType("NULL");
  }

  /**
   * Skips the next value in the JSON stream.
   *
   * @throws Error if there is no more JSON to read, or if the next token is a property name
   *
   * @example
   * // Skip properties you don't care about
   * await reader.beginObject();
   * while (await reader.hasNext()) {
   *   const name = await reader.nextName();
   *   if (name === "importantProperty") {
   *     const value = await reader.nextString();
   *     console.log(value);
   *   } else {
   *     await reader.skipValue(); // Skip this property
   *   }
   * }
   * await reader.endObject();
   */
  async skipValue() {
    let nextEvent = await this.peekNextEvent();
    if (nextEvent.type === "END_DOCUMENT") throw new Error('No more JSON');
    if (nextEvent.type === "NAME") throw new Error('At NAME, not able to skipValue');
    let count = 0;
    while (true) {
      if (nextEvent.type === 'BEGIN_ARRAY' || nextEvent.type === 'BEGIN_OBJECT') count++;
      if (nextEvent.type === 'END_ARRAY' || nextEvent.type === 'END_OBJECT') count--;
      this.nextEvent = undefined;
      if (count === 0) break;
      nextEvent = await this.peekNextEvent();
    }
  }

  /**
   * Reads the next complete JSON value (object, array, or primitive).
   *
   * This method parses and returns the entire next value in the JSON stream,
   * regardless of its complexity. It's useful when you want to read a complete
   * JSON structure without manually navigating through it.
   *
   * @returns A Promise that resolves to the parsed JSON value
   * @throws Error if the JSON is invalid or incomplete
   *
   * @example
   * // Read a complete JSON object
   * const result = await reader.nextJson();
   * console.log(result.content.name);
   */
  async nextJson() {
    const stack = [];
    while (true) {
      const { type: nextToken, value } = await this.peekNextEvent();
      let json;
      if (nextToken === "BEGIN_ARRAY") {
        stack.push({ value: [] });
      } else if (nextToken === "BEGIN_OBJECT") {
        stack.push({ value: {} });
      } else if (nextToken === "END_ARRAY" || nextToken === "END_OBJECT") {
        json = stack.pop().value;
      } else if (nextToken === "NAME") {
        const top = stack[stack.length - 1];
        top.name = value;
      } else {
        json = value;
      }
      this.nextEvent = undefined;
      if (json !== undefined) {
        if (stack.length === 0) {
          // ensure that garbage at end of document results in error
          await this.peekNextEvent();
          return json;
        }
        const top = stack[stack.length - 1];
        if (top.name !== undefined) {
          const obj = top.value;
          // Property named __proto__ requires special handling
          if (top.name !== '__proto__') obj[top.name] = json;else
          top.value = { ...obj, ['__proto__']: json };
        } else top.value.push(json);
        top.name = undefined;
      }
    }
  }
}

/**
 * Constants representing DOIP statuses and default operations
 */
class DoipConstants {

  /** OK: The operation was successfully processed. */
  static STATUS_OK = "0.DOIP/Status.001";
  /** Bad Request: The request was invalid in some way. */
  static STATUS_BAD_REQUEST = "0.DOIP/Status.101";
  /** Unauthenticated: The client did not successfully authenticate. */
  static STATUS_UNAUTHENTICATED = "0.DOIP/Status.102";
  /** Forbidden: The client successfully authenticated, but is unauthorized to invoke the operation. */
  static STATUS_FORBIDDEN = "0.DOIP/Status.103";
  /** Not Found: The digital object is not known to the service to exist. */
  static STATUS_NOT_FOUND = "0.DOIP/Status.104";
  /**
   * Conflict: The client tried to create a new digital object with an identifier
   * already in use by an existing digital object.
   */
  static STATUS_CONFLICT = "0.DOIP/Status.105";
  /** Declined: The service declines to execute the extended operation. */
  static STATUS_DECLINED = "0.DOIP/Status.200";
  /** System Error: Error other than the ones stated above occurred. */
  static STATUS_ERROR = "0.DOIP/Status.500";

  /**
   * This operation allows a client to get information about the DOIP service.
   * - Request attributes: none
   * - Input: empty
   * - Response attributes: none
   * - Output: the default serialization of the DOIP Service Information as a digital object.
   */
  static OP_HELLO = "0.DOIP/Op.Hello";
  /**
   * This operation requests the list of operations that can be invoked on the target object.
   * - Request attributes: none
   * - Input: none
   * - Response attributes: none
   * - Output: a serialized list of strings based on the default serialization, each of which
   *   is an operation id that the target object supports.
   */
  static OP_LIST_OPERATIONS = "0.DOIP/Op.ListOperations";
  /**
   * This operation creates a digital object within the DOIP service. The
   * target of a creation operation is the DOIP service itself.
   * - Request attributes: none
   * - Input: a serialized digital object. The "id" can be omitted to ask the DOIP service to
   *   automatically choose the id.
   * - Response attributes: none
   * - Output: the default serialization of the created object omitting element data.
   *   Notably, includes the identifier of the object (even if chosen by the client) and any
   *   changes to the object automatically performed by the DOIP service.
   */
  static OP_CREATE = "0.DOIP/Op.Create";
  /**
   * This operation retrieves (some parts of the) information represented by the target object.
   * - Request attributes:
   *      - "element": if specified, retrieves the data for that element
   *      - "includeElementData": if present, returns the serialization of the object
   *        including all element data
   * - Input: none
   * - Response attributes: none
   * - Output: the default output is the serialization of the object using the default
   *   serialization without element data. If "element" was specified, the output is a
   *   single bytes segment with the bytes of the specified element. If
   *   "includeElementData" was specified, the output is the full serialized digital object.
   */
  static OP_RETRIEVE = "0.DOIP/Op.Retrieve";
  /**
   * This operation updates (some parts of the) information represented by the target object.
   * - Request attributes: none
   * - Input: a serialized digital object. Elements which are not intended to be changed can
   *   be omitted from the input.
   * - Response attributes: none
   * - Output: the default serialization of the created object omitting element data.
   *   Notably, includes any changes to the object automatically performed by the DOIP
   *   service.
   */
  static OP_UPDATE = "0.DOIP/Op.Update";
  /**
   * This operation removes the target digital object from the management of the DOIP service.
   * - Request attributes: none
   * - Input: none
   * - Response attributes: none
   * - Output: none
   */
  static OP_DELETE = "0.DOIP/Op.Delete";
  /**
   * This operation is used to discover digital objects by searching metadata
   * contained in the set of digital objects managed by the DOIP service.
   * - Request attributes:
   *      - "query": the search query to be performed, in a textual representation
   *      - "pageNum": the page number to be returned, starting with 0
   *      - "pageSize": the page size to be returned; if missing or negative, all results will be
   *        returned; if zero, no results are returned, but the "size" is still returned
   *      - "sortFields": a comma-separated list of sort specifications, each of which is a field
   *        name optionally followed by ASC or DESC
   *      - "type": either "id", to return just object ids, or "full", to return full object data
   *        (omitting element data); defaults to "full"
   * - Input: none
   * - Response attributes: none
   * - Output: an object based on the default serialization with top-level properties:
   *      - "size": the number of results across all pages
   *      - "results": a list of results, each of which is either a string (the object id) or the
   *        default serialization of an object omitting element data.
   */
  static OP_SEARCH = "0.DOIP/Op.Search";
}

/**
 * Class representing a DOIP message as a series of {@link InDoipSegment}s, which is how messages are sent
 * in DOIP. When the input of a {@link DoipRequest} or the output of a {@link DoipResponse}
 * has multiple segments, an InDoipMessage can be used.
 *
 * For advanced usage. In most cases, the input of {@link DoipRequest} can be provided,
 * and the output of {@link DoipResponse} can be accessed, more simply as JSON or
 * one of the various JavaScript types for representing bytes.
 */
class InDoipMessage {
  [Symbol.asyncIterator]() {
    return this;
  }





  static async isEmpty(inDoipMessage) {
    for await (const segment of inDoipMessage) {
      return false;
    }
    return true;
  }

  static async getFirstSegment(inDoipMessage) {
    for await (const segment of inDoipMessage) {
      return segment;
    }
    return null;
  }
}

/**
 * Represents a segment of an {@link InDoipMessage}. Each segment may consist of JSON or arbitrary bytes.
 *
 * For advanced usage. In most cases, the input of {@link DoipRequest} can be provided,
 * and the output of {@link DoipResponse} can be accessed, more simply as JSON or
 * one of the various JavaScript types for representing bytes.
 */
class InDoipSegment {






  constructor(isJson, jsonValue, bodyValue) {
    if (isJson === undefined || isJson === null) {
      this.isJson = jsonValue !== undefined;
    } else {
      this.isJson = isJson;
    }
    if (jsonValue !== undefined) {
      this.blobValue = new Blob([JSON.stringify(jsonValue)]);
    } else if (bodyValue === null || bodyValue === undefined) {
      // the fetch API Body notion allows a null body to indicate an empty ReadableStream -- we use a non-null empty stream instead
      this.blobValue = new Blob([]);
    } else if (bodyValue instanceof ReadableStream) {
      this.readableStreamValue = bodyValue;
    } else if (bodyValue instanceof Blob) {
      this.blobValue = bodyValue;
    } else {
      this.blobValue = new Blob([bodyValue]);
    }
    this.response = new Response(this.blobValue || this.readableStreamValue);
  }

  static ofJson(isJson, json) {
    return new InDoipSegment(isJson, json, undefined);
  }

  static ofBytes(isJson, body) {
    return new InDoipSegment(isJson, undefined, body);
  }

  get body() {
    if (!this.readableStreamValue) this.readableStreamValue = this.response.body;
    return this.response.body;
  }

  get bodyUsed() {
    return this.response.bodyUsed;
  }

  async arrayBuffer() {
    return this.response.arrayBuffer();
  }

  async bytes() {
    return this.response.bytes();
  }

  async blob() {
    return this.response.blob();
  }

  async json() {
    return this.response.json();
  }

  async text() {
    return this.response.text();
  }

  async formData() {
    throw new TypeError('formData not supported');
  }

  /**
   * In situations where we can't just use ReadableStream -- namely, browsers which don't support ReadableStream requests in fetch --
   * we may want to know whether we had a ReadableStream or a Blob (in particular, a File) in order to construct a request
   * with the minimum of reading things into memory.
   */
  bytesAsProvided() {
    if (this.blobValue) return this.blobValue;
    return this.readableStreamValue;
  }

  close() {
    if (this.readableStreamValue) return this.readableStreamValue.cancel();
    return Promise.resolve();
  }
}

class InDoipMessageFromIterable extends InDoipMessage {


  constructor(iterable) {
    super();
    this.iterator = iterable[Symbol.iterator]();
  }

  async next() {
    return this.iterator.next();
  }

  async close() {
    while (true) {
      const res = await this.next();
      if (res.done) return;
      await res.value.close();
    }
  }
}

/**
 * This helper class can be used to create a new element with a body to
 * attach to a Digital Object.
 *
 * @example
 * ```javascript
 * const image = await fetch('https://www.cordra.org/assets/img/favicon.ico', { method: 'GET' });
 * const binaryElement = new ElementWithBody({
 *     id: "binaryElement",
 *     body: image.body,
 *     type: "image/vnd.microsoft.icon",
 *     attributes: {
 *         filename: "favicon.ico"
 *     }
 * });
 * const textFileElement = new ElementWithBody({
 *     id: "textFileElement",
 *     body: "This text body will be stored as a file called foo.txt",
 *     type: "application/text",
 *     attributes: {
 *         filename: "foo.txt"
 *     }
 * });
 * const digitalObject = {
 *     type: 'Document',
 *     elements: [ textFileElement, binaryElement ],
 *     attributes: {
 *         content: { name: 'Element Example' }
 *     }
 * };
 * const result = await client.create(digitalObject);
 * console.log(result.elements);
 * ```
 */
class ElementWithBody {






  /**
   * Constructs an ElementWithBody.
   *
   * @param element - The {@link Element}, with the usual "id", "type", "length", and "attributes" properties,
   *   and where the "body" property can be a ReadableStream<Uint8Array> or another way to provide the bytes
   *   (Blob, ArrayBuffer, ArrayBufferView, or string).
   */
  constructor(element) {
    this.id = element.id;
    this.type = element.type;
    this.length = element.length;
    this.attributes = element.attributes;
    if (element instanceof ElementWithBody) {
      this.bodySegment = InDoipSegment.ofBytes(false, element.bytesAsProvided());
    } else {
      this.bodySegment = InDoipSegment.ofBytes(false, element.body);
    }
  }

  get body() {
    return this.bodySegment.body;
  }

  get bodyUsed() {
    return this.bodySegment.bodyUsed;
  }

  async arrayBuffer() {
    return this.bodySegment.arrayBuffer();
  }

  async bytes() {
    return this.bodySegment.bytes();
  }

  async blob() {
    return this.bodySegment.blob();
  }

  async json() {
    return this.bodySegment.json();
  }

  async text() {
    return this.bodySegment.text();
  }

  async formData() {
    throw new TypeError('formData not supported');
  }

  /**
   * In situations where we can't just use ReadableStream -- namely, browsers which don't support ReadableStream requests in fetch --
   * we may want to know whether we had a ReadableStream or a Blob (in particular, a File) in order to construct a request
   * with the minimum of reading things into memory.
   */
  bytesAsProvided() {
    // eslint-disable-next-line @typescript-eslint/no-unnecessary-condition
    if (!this.bodySegment) throw new Error('No body for this Element');
    return this.bodySegment.bytesAsProvided();
  }
}

/* eslint-disable @typescript-eslint/no-explicit-any */


class ServiceInfoProvider {

  static handleProxyBaseUri = "https://hdl.handle.net/api/handles/";

  static DOIP_SERVICE = "0.TYPE/DOIPService";
  static DOIP_SERVICE_INFO = "0.TYPE/DOIPServiceInfo";

  async getServiceInfoForTarget(targetHandle) {
    try {
      const targetHandleRecord = await this.getHandleRecordFromProxy(targetHandle);
      let doipServiceInfoHandleValue = this.getValueByType(targetHandleRecord, ServiceInfoProvider.DOIP_SERVICE_INFO);
      if (!doipServiceInfoHandleValue) {
        const doipServiceHandleValue = this.getValueByType(targetHandleRecord, ServiceInfoProvider.DOIP_SERVICE);
        if (!doipServiceHandleValue) return null;
        const doipServiceHandle = doipServiceHandleValue.data.value;
        const doipServiceHandleRecord = await this.getHandleRecordFromProxy(doipServiceHandle);
        doipServiceInfoHandleValue = this.getValueByType(doipServiceHandleRecord, ServiceInfoProvider.DOIP_SERVICE_INFO);
      }

      if (!doipServiceInfoHandleValue) return null;
      const doipServiceInfoJson = doipServiceInfoHandleValue.data.value;
      const doipServiceInfoParsed = JSON.parse(doipServiceInfoJson);
      const serviceInfo = doipServiceInfoParsed.attributes;
      if (!serviceInfo.serviceId) serviceInfo.serviceId = doipServiceInfoParsed.id;
      return serviceInfo;
    } catch (e) {
      return null;
    }
  }

  async getHandleRecordFromProxy(handle) {
    const uri = ServiceInfoProvider.handleProxyBaseUri + encodeURIComponent(handle).replace(/%2F/g, '/');
    const httpResponse = await fetch(uri);
    const status = httpResponse.status;
    if (status === 200) {
      const handleRecord = await httpResponse.json();
      return handleRecord;
    } else {
      throw new Error("Unable to retrieve handle from proxy:" + handle);
    }
  }

  getValueByType(handleRecord, type) {
    for (const value of handleRecord.values) {
      if (type === value.type) {
        return value;
      }
    }
    return null;
  }
}

/**
 * This class implements the DoipClient interface and provides common functionality
 * for all DOIP client implementations.
 *
 * Concrete implementations need to provide the {@link performOperationViaRequest} method to
 * handle the actual communication with DOIP services, as well as a {@link close} method to
 * handle resource cleanup. This class will define all convenience methods by means of
 * `performOperationViaRequest`.
 */
class AbstractDoipClient {

  /**
   * Default options for all operations performed by this client.
   * These options can be overridden by providing options to individual method calls.
   */


  /**
   * Provider for DOIP service information.
   * Used to look up service information for targets when not explicitly provided.
   *
   * @internal
   */
  serviceInfoProvider = new ServiceInfoProvider();

  /**
   * Performs a DOIP operation.
   *
   * This method must be implemented by concrete subclasses to handle
   * the actual communication with DOIP services.
   *
   * @param doipRequest - The DOIP request to perform
   * @param options - Optional request options for this operation
   * @returns A promise that resolves to a DoipResponse
   */




  performOperation(targetIdOrDoipRequest, operationIdOrOptions, input, options) {
    if (typeof targetIdOrDoipRequest === 'string') {
      return this.performOperationViaRequest({ targetId: targetIdOrDoipRequest, operationId: operationIdOrOptions, input }, options);
    }
    return this.performOperationViaRequest(targetIdOrDoipRequest, operationIdOrOptions);
  }



  targetIdForService(options) {
    return options?.serviceInfo?.serviceId || this.defaultOptions?.serviceInfo?.serviceId || 'service';
  }

  /**
   * Determines the appropriate service information to use for a request.
   * First checks if service information is provided in the options, then falls back
   * to default options, and finally attempts to look up service information based on
   * the target ID.
   *
   * @param targetId - The ID of the target DOIP service
   * @param options - Optional request options
   * @returns A promise that resolves to the service information
   * @throws Error if no service information can be found
   * @internal
   */
  async getServiceInfoForRequest(targetId, options) {
    if (options?.serviceInfo) {
      return options.serviceInfo;
    }
    if (options?.lookupServiceInfo !== true) {
      if (this.defaultOptions?.serviceInfo) {
        return this.defaultOptions.serviceInfo;
      }
    }
    if (options?.lookupServiceInfo === false) {
      throw new Error("No service provided and lookupServiceInfo is false");
    }
    const serviceInfoFromTargetId = await this.serviceInfoProvider?.getServiceInfoForTarget(targetId);
    if (!serviceInfoFromTargetId) {
      throw new Error(`No service for target: ${targetId} could be found`);
    }
    return serviceInfoFromTargetId;
  }

  /**
   * This method creates a copy of the request and adds authentication information
   * and attributes from the options or default options if not already present.
   *
   * @param doipRequest - The DOIP request to augment
   * @param options - Optional request options
   * @returns The augmented DOIP request
   */
  augmentedRequest(doipRequest, options) {
    const res = { ...doipRequest };
    if (!res.authentication) {
      if (options?.authentication) {
        res.authentication = options.authentication;
      } else if (!options?.anonymous) {
        if (this.defaultOptions?.authentication) {
          res.authentication = this.defaultOptions.authentication;
        }
      }
    }
    if (options?.attributes || this.defaultOptions?.attributes) {
      res.attributes = { ...this.defaultOptions?.attributes, ...options?.attributes, ...res.attributes };
    }
    return res;
  }

  async hello(options) {
    let doipResponse;
    try {
      const doipRequest = {
        targetId: this.targetIdForService(options),
        operationId: DoipConstants.OP_HELLO
      };
      doipResponse = await this.performOperation(doipRequest, options);
      return await AbstractDoipClient.digitalObjectFromResponse(doipResponse);
    } finally {
      if (doipResponse) {
        await doipResponse.close();
      }
    }
  }

  async retrieve(targetId, options) {
    let doipResponse;
    try {
      const doipRequest = {
        targetId,
        operationId: DoipConstants.OP_RETRIEVE
      };
      doipResponse = await this.performOperation(doipRequest, options);
      if (doipResponse.getStatus() === DoipConstants.STATUS_NOT_FOUND) {
        return null;
      }
      return await AbstractDoipClient.digitalObjectFromResponse(doipResponse);
    } finally {
      if (doipResponse) {
        await doipResponse.close();
      }
    }
  }

  async listOperations(targetId, options) {
    const doipRequest = {
      targetId,
      operationId: DoipConstants.OP_LIST_OPERATIONS
    };
    const doipResponse = await this.performOperation(doipRequest, options);
    return doipResponse.asJson();
  }

  async searchIds(query, params, options) {
    return this.searchIdsOrFull("id", query, params, options);
  }

  async search(query, params, options) {
    return this.searchIdsOrFull("full", query, params, options);
  }

  async searchIdsOrFull(type, query, params, options) {
    const attributes = { ...params, query, type };
    const doipRequest = {
      targetId: this.targetIdForService(options),
      operationId: DoipConstants.OP_SEARCH,
      attributes
    };
    const doipResponse = await this.performOperation(doipRequest, options);
    return AbstractDoipClient.searchResultsFromResponse(doipResponse);
  }

  async delete(targetId, options) {
    let doipResponse;
    try {
      const doipRequest = {
        targetId,
        operationId: DoipConstants.OP_DELETE
      };
      doipResponse = await this.performOperation(doipRequest, options);
      if (doipResponse.getStatus() !== DoipConstants.STATUS_OK) {
        throw await doipResponse.asError();
      }
    } finally {
      if (doipResponse) {
        await doipResponse.close();
      }
    }
  }

  async retrieveElementBytes(targetId, elementId, options) {
    const attributes = {
      element: elementId
    };
    return this.retrieveElementBytesViaAttributes(targetId, attributes, options);
  }

  async retrievePartialElementBytes(targetId, elementId, start, end, options) {
    const attributes = {
      element: elementId,
      range: {
        start,
        end
      }
    };
    return this.retrieveElementBytesViaAttributes(targetId, attributes, options);
  }

  async retrieveElementBytesViaAttributes(targetId, attributes, options) {
    let doipResponse;
    let callerWillClose = false;
    try {
      const doipRequest = {
        targetId,
        operationId: DoipConstants.OP_RETRIEVE,
        attributes
      };
      doipResponse = await this.performOperation(doipRequest, options);
      if (doipResponse.getStatus() === DoipConstants.STATUS_NOT_FOUND) {
        return null;
      }
      callerWillClose = true;
      return await doipResponse.asBody();
    } finally {
      if (doipResponse && !callerWillClose) {
        await doipResponse.close();
      }
    }
  }

  async create(digitalObject, options) {
    const targetId = this.targetIdForService(options);
    const doipRequest = {
      targetId,
      operationId: DoipConstants.OP_CREATE
    };
    let doipResponse;
    try {
      if (AbstractDoipClient.hasElementStreamsOrBytes(digitalObject)) {
        const inDoipMessage = AbstractDoipClient.buildCreateOrUpdateMessageFrom(digitalObject, false);
        doipRequest.input = inDoipMessage;
        doipResponse = await this.performOperation(doipRequest, options);
      } else {
        const dobjClone = AbstractDoipClient.shallowCopyDigitalObjectMinusElementBytes(digitalObject);
        doipRequest.input = dobjClone;
        doipResponse = await this.performOperation(doipRequest, options);
      }
      return await AbstractDoipClient.digitalObjectFromResponse(doipResponse);
    } finally {
      if (doipResponse) {
        await doipResponse.close();
      }
    }
  }

  async update(digitalObject, options) {
    const targetId = digitalObject.id;
    const doipRequest = {
      targetId,
      operationId: DoipConstants.OP_UPDATE
    };
    let doipResponse;
    try {
      if (AbstractDoipClient.hasElementStreamsOrBytes(digitalObject)) {
        const inDoipMessage = AbstractDoipClient.buildCreateOrUpdateMessageFrom(digitalObject, true);
        doipRequest.input = inDoipMessage;
        doipResponse = await this.performOperation(doipRequest, options);
      } else {
        const dobjClone = AbstractDoipClient.shallowCopyDigitalObjectMinusElementBytes(digitalObject);
        doipRequest.input = dobjClone;
        doipResponse = await this.performOperation(doipRequest, options);
      }
      return await AbstractDoipClient.digitalObjectFromResponse(doipResponse);
    } finally {
      if (doipResponse) {
        await doipResponse.close();
      }
    }
  }

  static hasElementStreamsOrBytes(digitalObject) {
    if (digitalObject.elements) {
      for (const el of digitalObject.elements) {
        if (el.body) return true;
      }
    }
    return false;
  }

  static buildCreateOrUpdateMessageFrom(dobj, isUpdate) {
    const segments = [];
    const dobjClone = this.shallowCopyDigitalObjectMinusElementBytes(dobj);
    const dobjSegment = InDoipSegment.ofJson(true, dobjClone);
    segments.push(dobjSegment);
    if (dobj.elements) {
      for (const el of dobj.elements) {
        if (isUpdate && el.body === undefined) continue;
        const elementSegmentJson = { id: el.id };
        const elementHeaderSegment = InDoipSegment.ofJson(true, elementSegmentJson);
        segments.push(elementHeaderSegment);
        let bytesSegment;
        if (el.body === undefined) {
          throw new Error('Element without bytes used for create or update');
        } else if (el instanceof ElementWithBody) {
          bytesSegment = InDoipSegment.ofBytes(false, el.bytesAsProvided());
        } else {
          bytesSegment = InDoipSegment.ofBytes(false, el.body);
        }
        segments.push(bytesSegment);
      }
    }
    return new InDoipMessageFromIterable(segments);
  }

  static shallowCopyDigitalObjectMinusElementBytes(dobj) {
    const clone = {};
    clone.id = dobj.id;
    clone.type = dobj.type;
    clone.attributes = dobj.attributes;
    if (dobj.elements) {
      clone.elements = [];
      for (const el of dobj.elements) {
        const cloneEl = {
          id: el.id,
          type: el.type,
          length: el.length,
          attributes: el.attributes
        };
        clone.elements.push(cloneEl);
      }
    }
    return clone;
  }

  /**
   * Parses a DoipResponse into a DigitalObject, throwing an error if
   * the DoipResponse does not have the expected structure.
   *
   * @param doipResponse - The DOIP response to parse
   * @returns The DigitalObject given by the DOIP response
   */
  static async digitalObjectFromResponse(doipResponse) {
    try {
      if (doipResponse.getStatus() !== DoipConstants.STATUS_OK) {
        throw await doipResponse.asError();
      }
      const inDoipMessage = doipResponse.getOutput();
      const firstSegment = await InDoipMessage.getFirstSegment(inDoipMessage);
      if (firstSegment == null) {
        throw new Error("Missing input");
      }
      const digitalObject = {};
      Object.assign(digitalObject, await firstSegment.json());
      if (digitalObject.elements) {
        const elements = {};
        for (const el of digitalObject.elements) {
          elements[el.id] = el;
        }
        let elementId = null;
        for await (const segment of inDoipMessage) {
          if (elementId === null) {
            if (segment.isJson) {
              const json = await segment.json();
              if (json && typeof json === 'object' && 'id' in json) {
                elementId = json.id;
                if (typeof elementId === 'string') continue;
              }
            }
            throw new Error("Expected element header");
          } else {
            const elementBytesSegment = segment;
            const el = elements[elementId];
            if (!el) {
              throw new Error("No such element " + elementId);
            }
            elements[el.id] = new ElementWithBody({ ...el, body: elementBytesSegment.bytesAsProvided() });
            elementId = null;
          }
        }
        if (elementId !== null) {
          //Should have ended with a bytes segment
          throw new Error("Unexpected end of input");
        }
        digitalObject.elements = Object.values(elements);
      } else if (!(await InDoipMessage.isEmpty(inDoipMessage))) {
        throw new Error("Unexpected input segments");
      }
      return digitalObject;
    } finally {
      await doipResponse.close();
    }
  }

  /**
   * Parses a DoipResponse into a SearchResults, throwing an error if
   * the DoipResponse does not have the expected structure.
   *
   * @typeParam T The type of items in the search results. Either string or {@link DigitalObject}.
   * @param doipResponse - The DOIP response to parse
   * @returns The SearchResults given by the DOIP response
   */
  static async searchResultsFromResponse(doipResponse) {
    let callerWillClose = false;
    try {
      if (doipResponse.getStatus() !== DoipConstants.STATUS_OK) {
        throw await doipResponse.asError();
      }
      const jsonStream = await doipResponse.asReadableStream();
      callerWillClose = true;
      return await SearchResultsImpl.create(jsonStream);
    } finally {
      if (!callerWillClose) {
        await doipResponse.close();
      }
    }
  }
}

class SearchResultsImpl {

  size = -1;


  constructor(jsonStream) {
    this.jsonReader = new JsonReader(jsonStream);
  }

  async initialize() {
    await this.jsonReader.beginObject();
    while (await this.jsonReader.hasNext()) {
      const name = await this.jsonReader.nextName();
      if (name === 'size') this.size = await this.jsonReader.nextNumber();else
      if (name === 'facets') this.facets = await this.jsonReader.nextJson();else
      if (name === 'results') {
        await this.jsonReader.beginArray();
        break;
      }
    }
  }

  static async create(jsonStream) {
    const res = new SearchResultsImpl(jsonStream);
    await res.initialize();
    return res;
  }

  [Symbol.asyncIterator]() {
    return this;
  }

  async next() {
    if (!(await this.jsonReader.hasNext())) {
      await this.close();
      return { value: undefined, done: true };
    }
    const value = await this.jsonReader.nextJson();
    return { value };
  }

  async close() {
    return this.jsonReader.close();
  }
}

const webcrypto = globalThis.crypto;

/**
 * Provides functions for encoding and decoding data in Base64 format.
 * Supports both standard Base64 and URL-safe Base64.
 * @namespace
 */
const Base64 = { decode: decode$1, encode: encode$1, encodeUrlSafe };

const un$1 = undefined;

const base64DecodeArray = [
un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1,
un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1,
un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, un$1, 62, un$1, 62, un$1, 63,
52, 53, 54, 55, 56, 57, 58, 59, 60, 61, un$1, un$1, un$1, un$1, un$1, un$1,
// eslint-disable-next-line @stylistic/no-multi-spaces
un$1, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14,
15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, un$1, un$1, un$1, un$1, 63,
un$1, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51];


const base64EncodeArray = [
'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P',
'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f',
'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v',
'w', 'x', 'y', 'z', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '+', '/'];


const base64UrlEncodeArray = [
'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P',
'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', 'a', 'b', 'c', 'd', 'e', 'f',
'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v',
'w', 'x', 'y', 'z', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '-', '_'];


/**
 * Decodes a Base64 string into a byte array.
 *
 * @param s - The Base64 string to decode
 * @returns A Uint8Array containing the decoded bytes
 */
function decode$1(s) {
  const res = new Uint8Array(calcNumBytes$1(s));
  let pos = 0;
  let inFour = 0;
  let accum = 0;
  for (let i = 0; i < s.length; i++) {
    const code = base64DecodeArray[s.charCodeAt(i)];
    if (code === undefined) continue;
    if (inFour === 0) {
      accum = code << 2;
    } else if (inFour === 1) {
      accum |= code >>> 4;
      res[pos++] = accum;
      accum = (code & 0x0F) << 4;
    } else if (inFour === 2) {
      accum |= code >>> 2;
      res[pos++] = accum;
      accum = (code & 0x03) << 6;
    } else {
      accum |= code;
      res[pos++] = accum;
    }
    inFour = (inFour + 1) % 4;
  }
  return res.subarray(0, pos);
}

function calcNumBytes$1(str) {
  let len = str.length;
  // eslint-disable-next-line @typescript-eslint/prefer-string-starts-ends-with
  if (str[str.length - 1] === '=') {
    len -= 1;
    if (str[str.length - 2] === '=') len -= 1;
  }
  const mod = len % 4;
  if (mod === 0) return 3 * len / 4;
  if (mod === 1) return 3 * (len - 1) / 4;
  if (mod === 2) return 3 * (len - 2) / 4 + 1;
  return 3 * (len - 3) / 4 + 2;
}

function genericBase64EncoderFunction(arr, thisBase64EncodeArray, usePad) {
  let s = '';
  let accum = 0;
  let inThree = 0;
  for (const code of arr) {
    if (inThree === 0) {
      s += thisBase64EncodeArray[code >>> 2];
      accum = (code & 0x03) << 4;
    } else if (inThree === 1) {
      accum |= code >>> 4;
      s += thisBase64EncodeArray[accum];
      accum = (code & 0x0F) << 2;
    } else {
      accum |= code >>> 6;
      s += thisBase64EncodeArray[accum];
      s += thisBase64EncodeArray[code & 0x3F];
    }
    inThree = (inThree + 1) % 3;
  }
  if (inThree > 0) {
    s += thisBase64EncodeArray[accum];
    if (usePad) {
      const pad = '=';
      s += pad;
      if (inThree === 1) s += pad;
    }
  }
  return s;
}

/**
 * Encodes a byte array into a standard Base64 string, with padding.
 *
 * @param arr - The byte array to encode
 * @returns The Base64 encoded string
 */
function encode$1(arr) {
  return genericBase64EncoderFunction(arr, base64EncodeArray, true);
}

/**
 * Encodes a byte array into a URL-safe Base64 string.
 *
 * @param arr - The byte array to encode
 * @returns The URL-safe Base64 encoded string
 */
function encodeUrlSafe(arr) {
  return genericBase64EncoderFunction(arr, base64UrlEncodeArray, false);
}

/**
 * Provides functions for encoding and decoding data in hexadecimal format.
 * @namespace
 */
const Hex = { decode, encode };

const un = undefined;

const hexDecodeArray = [
un, un, un, un, un, un, un, un, un, un, un, un, un, un, un, un,
un, un, un, un, un, un, un, un, un, un, un, un, un, un, un, un,
un, un, un, un, un, un, un, un, un, un, un, un, un, un, un, un,
// eslint-disable-next-line @stylistic/no-multi-spaces
0, 1, 2, 3, 4, 5, 6, 7, 8, 9, un, un, un, un, un, un,
un, 10, 11, 12, 13, 14, 15, 16, un, un, un, un, un, un, un, un,
un, un, un, un, un, un, un, un, un, un, un, un, un, un, un, un,
un, 10, 11, 12, 13, 14, 15, 16];


const hexEncodeArray = [
'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'];


/**
 * Decodes a hexadecimal string into a byte array.
 *
 * @param str - The hexadecimal string to decode
 * @returns A Uint8Array containing the decoded bytes
 */
function decode(str) {
  const res = new Uint8Array(calcNumBytes(str));
  let pos = 0;
  let inTwo = 0;
  let accum = 0;
  for (let i = 0; i < str.length; i++) {
    const code = hexDecodeArray[str.charCodeAt(i)];
    if (code === undefined) continue;
    if (inTwo === 0) {
      accum = code << 4;
    } else {
      accum |= code;
      res[pos++] = accum;
    }
    inTwo = (inTwo + 1) % 2;
  }
  return res.subarray(0, pos);
}

function calcNumBytes(str) {
  return Math.floor(str.length / 2);
}

/**
 * Encodes a byte array into a hexadecimal string.
 * Uses uppercase letters (A-F) for values 10-15.
 *
 * @param arr - The byte array to encode
 * @returns The hexadecimal encoded string (uppercase)
 */
function encode(arr) {
  let s = '';
  for (const code of arr) {
    s += hexEncodeArray[code >>> 4];
    s += hexEncodeArray[code & 0x0F];
  }
  return s;
}

function getMessageFromErrorResponse(status, errorObject) {
  if (errorObject && typeof errorObject === 'object' && 'message' in errorObject && errorObject.message && typeof errorObject.message === 'string') {
    return errorObject.message;
  }
  return "DOIP Error: " + status;
}

/**
 * Represents a DOIP Error. This is an extension of the native JavaScript `Error` class.
 */
class DoipError extends Error {
  /**
   * The DOIP Status code
   * @example '0.DOIP/Status.200'
   */

  /**
   * The full response message. Generally a JSON object containing a detailed message.
   * @example { message: 'Operation not supported' }
   */


  constructor(status, response, options) {
    super(getMessageFromErrorResponse(status, response), options);
    this.name = "DoipError";
    this.status = status;
    this.response = response;
  }
}

const DelegatedCloseableReadableStream = {
  create
};

function create(readable, closeFunction) {
  const ts = new TransformStream();
  (async () => {
    try {
      try {
        await readable.pipeTo(ts.writable, { preventClose: true });
        await closeFunction();
        await ts.writable.close();
      } catch (reason) {
        await closeFunction();
        throw reason;
      }
    } catch (reason) {
      await Promise.all([
      readable.cancel(reason),
      ts.writable.abort(reason)]
      );
    }
  })().catch(() => {});
  return ts.readable;
}

/**
 * Represents a response from a DOIP operation.
 *
 * This class encapsulates the response data from a DOIP service, including status,
 * attributes, and output. It provides methods to access this data and convert
 * the response to different formats (JSON, ReadableStream, Body, or Error).
 *
 * Callers are generally responsible for calling {@link close} to ensure
 * the network connection behind this response is closed.
 * The {@link asJson}, {@link asBody}, {@link asReadableStream}, and {@link asError} methods
 * automatically close the response.
 */
class DoipResponse {




  constructor(initialSegmentJson, inDoipMessage) {
    this.initialSegmentJson = initialSegmentJson;
    this.inDoipMessage = inDoipMessage;
  }

  /**
   * Gets the DOIP status of the response
   * @example '0.DOIP/Status.001'
   */
  getStatus() {
    return this.initialSegmentJson.status;
  }

  /**
   * Gets all attributes from the response. The attributes that are
   * available depend on the operation that was invoked.
   * @returns The JSON object representing the attributes for this response
   */
  getAttributes() {
    return this.initialSegmentJson.attributes;
  }

  /**
   * Gets a specific attribute from the response
   * @returns The JSON associated with the given attribute, or undefined if not found
   */
  getAttribute(key) {
    return this.initialSegmentJson.attributes?.[key];
  }

  /**
   * Gets DOIP output as an InDoipMessage.
   *
   * This is a lower-level api for advanced usage. In most cases, {@link asJson},
   * {@link asBody}, and {@link asReadableStream} should be preferred.
   */
  getOutput() {
    if (this.initialSegmentJson.output) {
      return new InDoipMessageFromIterable([InDoipSegment.ofJson(true, this.initialSegmentJson.output)]);
    } else {
      return this.inDoipMessage;
    }
  }

  /**
   * Gets the response as a {@link DoipError} object. This allows easy access
   * to any error information in the response.
   * @returns A promise that resolves to a DoipError
   */
  async asError() {
    try {
      let errorMessage = this.getAttributes();
      const output = this.getOutput();
      // eslint-disable-next-line @typescript-eslint/no-unnecessary-condition
      if (output) {
        const firstSegment = await InDoipMessage.getFirstSegment(output);
        if (firstSegment && firstSegment.isJson) {
          errorMessage = await firstSegment.json();
        }
      }
      return new DoipError(this.getStatus(), errorMessage);
    } finally {
      await this.close();
    }
  }

  /**
   * Gets the response as a JSON object
   * @returns A promise that resolves to arbitrary JSON
   */
  async asJson() {
    try {
      if (this.getStatus() !== DoipConstants.STATUS_OK) {
        throw await this.asError();
      }
      const output = this.getOutput();
      // eslint-disable-next-line @typescript-eslint/no-unnecessary-condition
      if (!output) {
        throw new Error("Missing first segment in response");
      }
      const firstSegment = await InDoipMessage.getFirstSegment(output);
      if (!firstSegment) {
        throw new Error("Missing first segment in response");
      }
      return await firstSegment.json();
    } finally {
      await this.close();
    }
  }

  /**
   * Gets the response as a stream of bytes
   * @returns A promise that resolves to a ReadableStream
   */
  async asReadableStream() {
    let callerWillClose = false;
    try {
      if (this.getStatus() !== DoipConstants.STATUS_OK) {
        throw await this.asError();
      }
      const output = this.getOutput();
      // eslint-disable-next-line @typescript-eslint/no-unnecessary-condition
      if (!output) {
        throw new Error("Missing first segment in response");
      }
      const firstSegment = await InDoipMessage.getFirstSegment(output);
      if (!firstSegment) {
        throw new Error("Missing first segment in response");
      }
      callerWillClose = true;
      return DelegatedCloseableReadableStream.create(firstSegment.body, () => this.close());
    } finally {
      if (!callerWillClose) {
        await this.close();
      }
    }
  }

  /**
   * Get the response as a Body
   * @returns A promise that resolves to a Body
   */
  async asBody() {
    let callerWillClose = false;
    try {
      if (this.getStatus() !== DoipConstants.STATUS_OK) {
        throw await this.asError();
      }
      const inDoipMessage = this.getOutput();
      const firstSegment = await InDoipMessage.getFirstSegment(inDoipMessage);
      if (firstSegment == null) {
        throw new Error("Missing first segment in response");
      }
      const bytes = firstSegment.bytesAsProvided();
      if (bytes instanceof ReadableStream) {
        callerWillClose = true;
        return InDoipSegment.ofBytes(false, DelegatedCloseableReadableStream.create(bytes, async () => {
          await this.close();
        }));
      } else {
        return firstSegment;
      }
    } finally {
      if (!callerWillClose) {
        await this.close();
      }
    }
  }

  /**
   * Close the response object and clean up resources, if needed
   */
  async close() {
    // eslint-disable-next-line @typescript-eslint/no-unnecessary-condition
    if (this.inDoipMessage) {
      await this.inDoipMessage.close();
    }
  }
}

/**
 * Used by a DoipRequest to authenticate to a DOIP service.
 */
class AuthenticationInfo {}

const REQUEST_ID_LENGTH = 32;
const REQUEST_ID_CHARS = '0123456789ABCDEF';

function isCompact(input) {
  if (input === undefined) return true;
  if (input instanceof InDoipMessage) return false;
  if (input instanceof ReadableStream) return false;
  if (input instanceof Blob) return false;
  if (input instanceof ArrayBuffer) return false;
  if (ArrayBuffer.isView(input)) return false;
  return true;
}

async function headersFrom(doipRequest) {
  const doipRequestHeaders = {};
  if (!doipRequest.requestId) {
    doipRequestHeaders.requestId = generateRequestId();
  } else {
    doipRequestHeaders.requestId = doipRequest.requestId;
  }
  doipRequestHeaders.targetId = doipRequest.targetId;
  doipRequestHeaders.operationId = doipRequest.operationId;
  doipRequestHeaders.clientId = doipRequest.clientId;
  if (doipRequest.authentication instanceof AuthenticationInfo) {
    if (!doipRequest.clientId) doipRequestHeaders.clientId = doipRequest.authentication.getClientId();
    doipRequestHeaders.authentication = await doipRequest.authentication.getAuthentication();
    // if (doipRequest.authentication instanceof PasswordAuthenticationInfo) {
    //     doipRequestHeaders.clientId = undefined;
    // }
  } else if (doipRequest.authentication) {
    doipRequestHeaders.authentication = doipRequest.authentication;
  }
  if (doipRequest.attributes) {
    doipRequestHeaders.attributes = doipRequest.attributes;
  } else {
    doipRequestHeaders.attributes = {};
  }
  if (doipRequest.input !== undefined && isCompact(doipRequest.input)) {
    doipRequestHeaders.input = doipRequest.input;
  }
  return doipRequestHeaders;
}

function generateRequestId() {
  let result = '';
  for (let i = REQUEST_ID_LENGTH; i > 0; i--) {
    result += REQUEST_ID_CHARS[Math.floor(Math.random() * REQUEST_ID_CHARS.length)];
  }
  return result;
}

function messageOfInput(input) {
  if (input === undefined) return undefined;
  if (input instanceof InDoipMessage) return input;
  if (input instanceof ReadableStream ||
  input instanceof Blob ||
  input instanceof ArrayBuffer ||
  ArrayBuffer.isView(input)) {
    const inputTyped = input;
    return new InDoipMessageFromIterable([InDoipSegment.ofBytes(false, inputTyped)]);
  }
  return new InDoipMessageFromIterable([InDoipSegment.ofJson(true, input)]);
}

var DoipRequestUtil = {
  generateRequestId,
  isCompact,
  headersFrom,
  messageOfInput
};

class InDoipMessageFromHttpResponse extends InDoipMessage {





  constructor(response) {
    super();
    this.response = response;
    this.done = false;
    this.isJson = this.isJsonResponse();
  }

  isJsonResponse() {
    // TODO handle multipart responses
    const type = this.response.headers.get("Content-Type");
    if (!type) return false;
    if ("application/json" === type) return true;
    if (!type.startsWith("application/json")) return false;
    const nextChar = type.charAt("application/json".length);
    if (nextChar === ' ' || nextChar === '\t' || nextChar === ';') return true;
    return false;
  }

  async next() {
    if (this.done) return { done: true, value: undefined };
    this.done = true;
    const segment = InDoipSegment.ofBytes(this.isJson, this.response.body);
    return {
      done: false,
      value: segment
    };
  }

  async close() {
    if (!this.done) await this.response.body?.cancel();
    this.done = true;
  }
}

// Note: since streaming requests is only supported in certain browsers (late 2023) and only for HTTP/2, we use Blob exclusively
// // https://developer.chrome.com/articles/fetch-streaming-requests/
// const supportsRequestStreams = (() => {
//     let duplexAccessed = false;

//     const hasContentType = new Request('https://example.org', {
//         body: new ReadableStream(),
//         method: 'POST',
//         get duplex() {
//             duplexAccessed = true;
//             return 'half';
//         }
//     } as RequestInit).headers.has('Content-Type');

//     return duplexAccessed && !hasContentType;
// })();

const textEncoder$1 = new TextEncoder();

/**
 * DoipClient implementation which uses the DOIP API for HTTP Clients.
 * See also [Cordra's documentation on this API](https://www.cordra.org/documentation/api/doip-api-for-http-clients.html).
 * Prefer {@link StandardDoipClient}.
 *
 * @internal
 */
class HttpDoipClient extends AbstractDoipClient {



  constructor(baseUriOrOptions, optionsParam) {
    super();
    if (!baseUriOrOptions) return;
    let options;
    if (typeof baseUriOrOptions === 'string') {
      options = { ...optionsParam, baseUri: baseUriOrOptions };
    } else {
      options = { ...baseUriOrOptions };
    }
    if (options.baseUri) {
      if (!options.serviceInfo) {
        options.serviceInfo = { baseUri: options.baseUri };
      } else {
        options.serviceInfo = this.buildServiceInfo(options.baseUri, options.serviceInfo);
      }
      delete options.baseUri;
    }
    this.defaultOptions = options;
  }

  buildServiceInfo(baseUri, serviceInfo) {
    const res = { ...serviceInfo };
    if (serviceInfo.interfaces) {
      res.interfaces = [...serviceInfo.interfaces];
    } else {
      res.interfaces = [];
    }
    res.interfaces.push({ baseUri });
    return res;
  }

  static baseUriForServiceInfo(serviceInfo) {
    const interfaces = serviceInfo.interfaces ? [serviceInfo, ...serviceInfo.interfaces] : [serviceInfo];
    for (const iface of interfaces) {
      if (iface.baseUri) return iface.baseUri;
    }
    for (const iface of interfaces) {
      if (iface.protocol === "HTTPS" && iface.ipAddress && iface.port) {
        return "https://" + iface.ipAddress + ":" + iface.port + (iface.path || "");
      }
    }
    throw new Error("No HTTPS interface available");
  }

  async performOperationViaRequest(doipRequest, options) {
    doipRequest = this.augmentedRequest(doipRequest, options);
    const doipRequestHeaders = await DoipRequestUtil.headersFrom(doipRequest);
    const serviceInfoForRequest = await this.getServiceInfoForRequest(doipRequest.targetId, options);
    const baseUriForRequest = HttpDoipClient.baseUriForServiceInfo(serviceInfoForRequest);
    if (DoipRequestUtil.isCompact(doipRequest.input)) {
      const doipResponse = await this.sendCompactRequest(doipRequestHeaders, baseUriForRequest);
      return doipResponse;
    }
    const inDoipMessage = DoipRequestUtil.messageOfInput(doipRequest.input);
    const doipResponse = await this.sendMultiPartRequest(doipRequestHeaders, inDoipMessage, baseUriForRequest);
    return doipResponse;
  }

  async sendMultiPartRequest(doipRequestHeaders, inDoipMessageRequest, baseUriForRequest) {
    const url = this.buildUriForHeaders(doipRequestHeaders, baseUriForRequest);
    const blobParts = [];
    const boundary = webcrypto.randomUUID();
    const headers = {
      "Content-Type": "multipart/mixed; boundary=" + boundary
    };
    if (doipRequestHeaders.authentication) {
      headers.Authorization = this.buildDoipAuthHeader(doipRequestHeaders.authentication);
    }
    let count = 0;
    for await (const segment of inDoipMessageRequest) {
      const defaultName = "part-" + count; //At time of writing server side code requires part names for multipart/mixed
      count++;
      const partContentType = segment.isJson ? "application/json" : "application/octet-stream";
      const boundaryHeaders = this.createBoundaryHeadersString(boundary, partContentType, defaultName);
      blobParts.push(boundaryHeaders);
      blobParts.push(await segment.blob());
    }
    const finalBoundary = this.createFinalBoundaryString(boundary);
    blobParts.push(finalBoundary);
    try {
      const response = await fetch(url, {
        method: 'POST',
        headers,
        body: new Blob(blobParts)
      });
      return this.buildDoipResponseForHttpResponse(response);
    } catch (e) {
      console.log(e);
      throw e;
    }
  }

  createBoundaryHeadersString(boundary, contentType, name) {
    let result = "\r\n--" + boundary + "\r\n";
    result += "Content-Disposition: form-data; name=" + name + "\r\n";
    result += "Content-Type: " + contentType + "\r\n\r\n";
    return result;
  }

  createFinalBoundaryString(boundary) {
    return "\r\n--" + boundary + "--\r\n\r\n";
  }

  async sendCompactRequest(doipRequestHeaders, baseUriForRequest) {
    const httpRequest = {
      method: "POST",
      headers: {}
    };
    const url = this.buildUriForHeaders(doipRequestHeaders, baseUriForRequest);
    if (doipRequestHeaders.authentication) {
      this.addCredentials(httpRequest, doipRequestHeaders.authentication);
    }
    if (doipRequestHeaders.input) {
      httpRequest.headers = {
        "Content-Type": "application/json"
      };
      httpRequest.body = JSON.stringify(doipRequestHeaders.input);
    }
    if (doipRequestHeaders.authentication) {
      httpRequest.headers.Authorization = this.buildDoipAuthHeader(doipRequestHeaders.authentication);
    }
    try {
      const response = await fetch(url, httpRequest);
      return this.buildDoipResponseForHttpResponse(response);
    } catch (e) {
      console.log(e);
      throw e;
    }
  }

  buildDoipResponseForHttpResponse(response) {
    const initialSegmentJson = this.initialSegmentJsonFromHttpResponse(response);
    const inDoipMessage = new InDoipMessageFromHttpResponse(response);
    const doipResponse = new DoipResponse(initialSegmentJson, inDoipMessage);
    return doipResponse;
  }

  //Only used if the doip response header is missing
  static statusMap = new Map([
  [200, DoipConstants.STATUS_OK],
  [400, DoipConstants.STATUS_BAD_REQUEST],
  [401, DoipConstants.STATUS_UNAUTHENTICATED],
  [403, DoipConstants.STATUS_FORBIDDEN],
  [404, DoipConstants.STATUS_NOT_FOUND],
  [409, DoipConstants.STATUS_CONFLICT],
  [500, DoipConstants.STATUS_ERROR]]
  );

  initialSegmentJsonFromHttpResponse(response) {
    const doipResponseHeaderValue = response.headers.get("Doip-Response");
    if (doipResponseHeaderValue != null) {
      return JSON.parse(doipResponseHeaderValue);
    } else {
      const status = response.status;
      const doipStatus = HttpDoipClient.statusMap.get(status);
      return {
        status: doipStatus
      };
    }
  }

  buildUriForHeaders(headers, baseUriForRequest) {
    let uri = baseUriForRequest +
    "?operationId=" + HttpDoipClient.getEncodedWithSlashes(headers.operationId) +
    "&targetId=" + HttpDoipClient.getEncodedWithSlashes(headers.targetId);
    if (headers.clientId != null) {
      uri += "&clientId=" + HttpDoipClient.getEncodedWithSlashes(headers.clientId);
    }
    if (headers.attributes != null) {
      const attributesJson = JSON.stringify(headers.attributes);
      uri += "&attributes=" + HttpDoipClient.getEncodedWithSlashes(attributesJson);
    }
    return uri;
  }

  static getEncodedWithSlashes(toEncode) {
    return encodeURIComponent(toEncode).replace(/%2F/g, '/');
  }

  addCredentials(httpRequest, authentication) {
    this.addDoipAuthHeader(httpRequest.headers, authentication);
  }

  addDoipAuthHeader(headers, authentication) {
    headers.Authorization = this.buildDoipAuthHeader(authentication);
  }

  buildDoipAuthHeader(authentication) {
    const jsonString = JSON.stringify(authentication);
    const base64OfJson = Base64.encode(textEncoder$1.encode(jsonString));
    return "Doip " + base64OfJson;
  }

  async close() {

    //no-op
  }}

var NativeDoipClient = undefined;

// eslint-disable-next-line @typescript-eslint/no-unnecessary-condition
const nativeSupport = NativeDoipClient !== undefined;

/**
 * This class provides an implementation of the DOIP client that can use
 * either HTTPS or native TCP for DOIP communication, depending on the available
 * interfaces and configuration. It automatically selects the appropriate
 * protocol (HTTPS or TCP) based on the service information and client capabilities.
 *
 * @example
 * ```javascript
 * // Create a client with service info and authentication
 * const authentication =
 *      new PasswordAuthenticationInfo('admin', 'password');
 * const serviceInfo = {
 *     ipAddress: 'localhost',
 *     port: 9000
 * };
 * const client = new StandardDoipClient({ serviceInfo, authentication });
 *
 * // Create a client with a base URI
 * const client = new StandardDoipClient('https://localhost:8443/doip');
 *
 * // Create a client with multiple interfaces and a preferred protocol
 * const serviceInfo = {
 *     interfaces: [
 *         { ipAddress: '127.0.0.1', port: 9000 },
 *         { baseUri: 'https://localhost:8443/doip/' }
 *     ]
 * };
 * const client = new StandardDoipClient({ serviceInfo, preferredProtocol: 'TCP' });
 * ```
 */
class StandardDoipClient extends AbstractDoipClient {




  /**
   * Constructs a StandardDoipClient.
   *
   * @param options - Optional default request options.
   *   The serviceInfo and authentication property can be used to provide a default serviceInfo and authentication for all requests.
   *   The options may optionally specify a "preferredProtocol", either "TCP" for native DOIP or "HTTPS" for the DOIP API for HTTP Clients.
   *   For convenience the options may specify a "baseUri", the base HTTP endpoint of a DOIP API for HTTP Clients service.
   */

  /**
   * Convenience constructor for a StandardDoipClient which uses the DOIP API for HTTP Clients.
   *
   * @param baseUri - The base HTTP endpoint of the DOIP API for HTTP Clients service
   * @param options - Optional default request options.
   *   The authentication property can be used to provide a default authentication for all requests.
   */

  constructor(baseUriOrOptions, optionsParam) {
    super();
    if (!baseUriOrOptions) return;
    let options;
    if (typeof baseUriOrOptions === 'string') {
      options = { ...optionsParam, baseUri: baseUriOrOptions };
    } else {
      options = { ...baseUriOrOptions };
    }
    if (options.baseUri) {
      if (!options.serviceInfo) {
        options.serviceInfo = { baseUri: options.baseUri };
      } else {
        options.serviceInfo = this.buildServiceInfo(options.baseUri, options.serviceInfo);
      }
      delete options.baseUri;
    }
    if (options.preferredProtocol) {
      this.preferredProtocol = options.preferredProtocol;
      delete options.preferredProtocol;
    }
    this.defaultOptions = options;
  }

  buildServiceInfo(baseUri, serviceInfo) {
    const res = { ...serviceInfo };
    if (serviceInfo.interfaces) {
      res.interfaces = [...serviceInfo.interfaces];
    } else {
      res.interfaces = [];
    }
    res.interfaces.push({ baseUri });
    return res;
  }

  selectServiceInfoInterface(serviceInfo) {
    const interfaces = serviceInfo.interfaces ? [serviceInfo, ...serviceInfo.interfaces] : [serviceInfo];
    if (this.preferredProtocol === "TCP") {
      for (const iface of interfaces) {
        if (this.isAcceptableInterface(iface, "TCP")) return iface;
      }
    } else if (this.preferredProtocol === "HTTPS") {
      for (const iface of interfaces) {
        if (iface.baseUri) return iface;
      }
      for (const iface of interfaces) {
        if (this.isAcceptableInterface(iface, "HTTPS")) return iface;
      }
    }
    for (const iface of interfaces) {
      if (this.isAcceptableInterface(iface, undefined)) return iface;
    }
    let message = "No usable DOIP interface available";
    if (this.hasAcceptableInterfaceIfNativeSupport(interfaces)) {
      message += " (no native DOIP support in this client; an HTTP interface is needed)";
    }
    throw new Error(message);
  }

  isAcceptableInterface(iface, preferredProtocol) {
    if ((!iface.protocol || iface.protocol === "TCP") && iface.ipAddress && iface.port) {
      if (!preferredProtocol || preferredProtocol === "TCP") return nativeSupport;
    }
    if (iface.baseUri) {
      if (!preferredProtocol || preferredProtocol === "HTTPS") return true;
    }
    if (iface.protocol === "HTTPS" && iface.ipAddress && iface.port) {
      if (!preferredProtocol || preferredProtocol === "HTTPS") return true;
    }
    return false;
  }

  isAcceptableInterfaceIfNativeSupport(iface) {
    if ((!iface.protocol || iface.protocol === "TCP") && iface.ipAddress && iface.port) {
      return true;
    }
    return false;
  }

  hasAcceptableInterfaceIfNativeSupport(interfaces) {
    for (const iface of interfaces) {
      if (this.isAcceptableInterfaceIfNativeSupport(iface)) {
        return true;
      }
    }
    return false;
  }

  shouldUseNative(iface) {
    return false;
  }

  async performOperationViaRequest(doipRequest, options) {
    doipRequest = this.augmentedRequest(doipRequest, options);
    const serviceInfoForRequest = await this.getServiceInfoForRequest(doipRequest.targetId, options);
    const iface = this.selectServiceInfoInterface(serviceInfoForRequest);
    if (this.shouldUseNative(iface)) {
      if (!this.nativeDoipClient) {
        this.nativeDoipClient = new NativeDoipClient(this.defaultOptions);
      }
      return this.nativeDoipClient.performOperation(doipRequest, options);
    } else {
      if (!this.httpDoipClient) {
        this.httpDoipClient = new HttpDoipClient(this.defaultOptions);
      }
      return this.httpDoipClient.performOperation(doipRequest, options);
    }
  }

  async close() {
    return Promise.all([
    this.httpDoipClient?.close(),
    this.nativeDoipClient?.close()]
    ).then();
  }
}

/**
 * Gets an attribute from a DigitalObject or Element.
 *
 * @param obj - The DigitalObject or Element
 * @param attr - The name of the attribute to retrieve
 * @returns The value of the attribute, or undefined if missing
 */
function getAttribute(obj, attr) {
  return obj.attributes?.[attr];
}

/**
 * Sets a top-level attribute on a DigitalObject or Element.
 *
 * @param obj - The DigitalObject or Element
 * @param attr - The name of the attribute to set
 * @param value - The value to set the attribute to
 */
function setAttribute(obj, attr, value) {
  obj.attributes ??= {};
  obj.attributes[attr] = value;
}

/**
 * Gets an element from a DigitalObject by the element id.
 *
 * @param obj - The DigitalObject
 * @param id - The identifier of the Element
 * @returns The Element, or undefined if missing
 * @function
 */
function getElementById(obj, id) {
  return obj.elements?.find((el) => el.id === id);
}

/**
 * Convenience functions for manipulating DigitalObjects and Elements.
 * @class
 */
var DigitalObjectUtil = {
  getAttribute,
  setAttribute,
  getElementById
};

/**
 * Fully reads a SearchResults, returning (asynchronously) the whole results as JSON.
 *
 * @typeParam T The type of items in the search results. Either string or {@link DigitalObject}.
 * @param searchResults - The search results
 * @returns The search results as JSON with a property "results" which is a JSON array.
 * @function
 */
async function searchResultsToJson(searchResults) {
  const res = {
    size: searchResults.size
  };
  if (searchResults.facets) res.facets = searchResults.facets;
  const results = [];
  for await (const item of searchResults) {
    results.push(item);
  }
  res.results = results;
  return res;
}

/**
 * Convenience functions for manipulating SearchResults.
 * @class
 */
var SearchUtil = {
  searchResultsToJson
};

/**
 * Used for password-based authentication.
 *
 * @example
 * ```javascript
 * const authentication = new PasswordAuthenticationInfo('admin', 'password');
 * const serviceInfo = {
 *     ipAddress: 'localhost',
 *     port: 9000
 * };
 * const client = new StandardDoipClient({ authentication, serviceInfo });
 * ```
 */
class PasswordAuthenticationInfo extends AuthenticationInfo {






  /**
   * Constructs a PasswordAuthenticationInfo.
   *
   * @param username - Username to use for authentication
   * @param password - Password to use for authentication
   * @param asUserId - If present, instructs the DOIP service to perform operations as this user
   * @param clientId - Optional clientId to send in requests using this token
   */
  constructor(username, password, asUserId, clientId) {
    super();
    this.username = username;
    this.password = password;
    this.asUserId = asUserId;
    this.clientId = clientId;
  }

  async getAuthentication() {
    return this;
  }

  getClientId() {return undefined;}
}

const textEncoder = new TextEncoder();

/**
 * Used for authenticating using a private key.
 *
 * @example
 * ```javascript
 * const clientId = '20.500.123/2';
 * const jwk = {
 *     kty: "RSA",
 *     n: "k31...",
 *     e: "AQAB",
 *     d: "NEY...",
 *     p: "vR9...",
 *     q: "x6I...",
 *     dp: "plB...",
 *     dq: "oqp...",
 *     qi: "s7M..."
 * };
 * const authentication = new PrivateKeyAuthenticationInfo(clientId, jwk);
 * const serviceInfo = {
 *     ipAddress: 'localhost',
 *     port: 9000
 * };
 * const client = new StandardDoipClient({ authentication, serviceInfo });
 * ```
 */
class PrivateKeyAuthenticationInfo extends AuthenticationInfo {





  /**
   * Constructs a PrivateKeyAuthenticationInfo.
   *
   * @param clientId - Client ID for this private key
   * @param privateKey - Private key to use to generate authentication for requests, in JSON Web Key format
   * @param asUserId - If present, instructs the DOIP service to perform operations as this user
   */
  constructor(clientId, privateKey, asUserId) {
    super();
    this.clientId = clientId;
    this.privateKey = privateKey;
    this.asUserId = asUserId;
  }

  getClientId() {
    return this.clientId;
  }

  getPrivateKey() {
    return this.privateKey;
  }

  async getAuthentication() {
    const token = await this.createBearerToken();
    const authentication = {
      token
    };
    if (this.asUserId) authentication.asUserId = this.asUserId;
    return authentication;
  }

  async createBearerToken() {
    const format = 'jwk';
    const extractable = false;
    const algo = {
      name: 'RSASSA-PKCS1-v1_5',
      hash: { name: 'SHA-256' }
    };
    const jwtHeader = { alg: 'RS256' };
    const jwtHeaderJson = JSON.stringify(jwtHeader);
    const jwtHeaderJsonBytes = textEncoder.encode(jwtHeaderJson);
    const jwtHeaderJsonBase64 = Base64.encodeUrlSafe(jwtHeaderJsonBytes);

    const nowSeconds = Math.floor(Date.now() / 1000);
    const claims = {
      iss: this.clientId,
      sub: this.clientId,
      //jti: this.generateJti(),
      iat: nowSeconds,
      exp: nowSeconds + 600
    };
    const claimsJson = JSON.stringify(claims);
    const claimsJsonBytes = textEncoder.encode(claimsJson);
    const claimsJsonBase64 = Base64.encodeUrlSafe(claimsJsonBytes);

    const thingToBeSigned = textEncoder.encode(jwtHeaderJsonBase64 + '.' + claimsJsonBase64);

    const resultKey = await webcrypto.subtle.importKey(format, this.privateKey, algo, extractable, ['sign']);
    const signature = await webcrypto.subtle.sign(algo.name, resultKey, thingToBeSigned);
    const sigAsString = Base64.encodeUrlSafe(new Uint8Array(signature));
    return jwtHeaderJsonBase64 + '.' + claimsJsonBase64 + '.' + sigAsString;
  }

  generateJti() {
    const bytes = webcrypto.getRandomValues(new Uint8Array(10));
    return Hex.encode(bytes);
  }
}

/**
 * Used for token-based authentication.
 *
 * @example
 * ```javascript
 * // Acquire an access token using the method called for by your DOIP service
 * const authentication = new TokenAuthenticationInfo('ACCESS_TOKEN');
 * const serviceInfo = {
 *     ipAddress: 'localhost',
 *     port: 9000
 * };
 * const client = new StandardDoipClient({ authentication, serviceInfo });
 * ```
 */
class TokenAuthenticationInfo extends AuthenticationInfo {

  /** Optional Client ID or username for this token. */

  /**
   * Authentication token. This should be acquired from the DOIP service using whatever
   * method is provided by that service.
   */

  /** If present, operations will be preformed as this user. */


  /**
   * Constructs a TokenAuthenticationInfo.
   *
   * @param token - Authentication token.
   *   This should be acquired from the DOIP service using whatever method is provided by that service.
   * @param clientId - Optional clientId to send in requests using this token
   * @param asUserId - If present, instructs the DOIP service to perform operations as this user
   */
  constructor(token, clientId, asUserId) {
    super();
    this.token = token;
    this.clientId = clientId;
    this.asUserId = asUserId;
  }

  getClientId() {
    return this.clientId;
  }

  /**
   * Returns the authentication token.
   */
  getToken() {
    return this.token;
  }

  async getAuthentication() {
    const authentication = {
      token: this.token
    };
    if (this.asUserId) authentication.asUserId = this.asUserId;
    return authentication;
  }
}

class InDoipMessageFromAsyncIterable extends InDoipMessage {


  constructor(asyncIterable) {
    super();
    this.asyncIterator = asyncIterable[Symbol.asyncIterator]();
  }

  async next() {
    return this.asyncIterator.next();
  }

  async close() {
    while (true) {
      const res = await this.next();
      if (res.done) return;
      await res.value.close();
    }
  }
}

export { AbstractDoipClient, AuthenticationInfo, DigitalObjectUtil, DoipConstants, DoipError, DoipRequestUtil, DoipResponse, ElementWithBody, HttpDoipClient, InDoipMessage, InDoipMessageFromAsyncIterable, InDoipMessageFromIterable, InDoipSegment, PasswordAuthenticationInfo, PrivateKeyAuthenticationInfo, SearchUtil, StandardDoipClient, TokenAuthenticationInfo };
//# sourceMappingURL=doip-client-1.1.0.js.map
