# DOIP Client Library - JavaScript Version (v1.1.0)

A JS client library for the Digital Object Interface Protocol (DOIP) usable in either browser or Node.

This DOIP Client Library can be used to develop JavaScript and TypeScript applications that are based on DOIP. It provides a set of classes for interacting with DOIP servers, including methods for making requests and handling responses.

The client library is written in TypeScript; type definitions are included in the release distribution for use in your applications.

## Requirements

Since this is a client library, you will need a DOIP service to connect to. The library itself does not provide a DOIP service. For testing purposes, you can use a local service that implements DOIP, such as [Cordra](https://www.cordra.org/download.html). Setting up such a service is beyond the scope of this README.

The library requires either a browser or Node.js JavaScript environment.

## Installation

### Node

Install this library via npm:

    npm install @cnri/doip-client

### Browser

The browser library is distributed as an ESM module. To use it, first, <a href="https://www.cordra.org/doip.html">download the bundle</a> and copy the file `doip-client-1.1.0.js` to a location accessible by your web server. Then, you can use the library by importing it in a script tag:

    <script type="module">
        import { StandardDoipClient } from "path/to/doip-client-1.1.0.js";

        const client = new StandardDoipClient(...);
    </script>

## Usage

Create a client instance using the DOIP native TCP interface:

```javascript
import { StandardDoipClient } from '@cnri/doip-client';

const serviceInfo = {
  ipAddress: 'localhost',
  port: 9000
};
const client = new StandardDoipClient({ serviceInfo });
```

The DOIP API for HTTP Clients is also supported.

Here is an example of searching a Cordra server for all Schema objects, printing out their IDs and names:

```javascript
const results = await client.search('type:Schema');
console.log('Number of results: ' + results.size);
for await (const result of results) {
  console.log(`${result.id}: ${result.attributes.content.name}`);
}
```

Here is an example of retrieving a specific object by ID:

```javascript
const result = await client.retrieve('20.500.123/1');
console.log(JSON.stringify(result.attributes.content, null, 2));
```

Here is an example of creating, updating, and deleting an object. The client is created with a username and password which is used for each request:

```javascript
const authentication = new PasswordAuthenticationInfo('admin', 'INSERT_PASSWORD');
const serviceInfo = {
  ipAddress: 'localhost',
  port: 9000
};
const client = new StandardDoipClient({ serviceInfo, authentication });

let object = {
  type: 'Document',
  attributes: {
    content: {
      name: 'Example Document'
    }
  }
};
object = await client.create(object);
console.log(object.id);
object.attributes.content.name = 'Updated Example Document';
object = await client.update(object);
console.log(object.attributes.content.name);
await client.delete(object.id);
const results = await client.retrieve(object.id);
console.log(results ? 'Error: Object found after deletion' : 'Delete successful');
client.close();
```

For more information on how to use this library, see the [API documentation](https://www.cordra.org/doip/doip-client-js/api-docs/index.html). You can also read the [DOIP Specification](https://www.dona.net/sites/default/files/2018-11/DOIPv2Spec_1.pdf) for more information about DOIP and its various components; Cordra documentation also includes sections on [DOIP](https://www.cordra.org/documentation/api/doip.html) and [the DOIP API for HTTP Clients](https://www.cordra.org/documentation/api/doip-api-for-http-clients.html).
